function Mv = movieGBNS(x, T, U, V, M, Nframes, FigNumber)
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% function Mv = movieGBNS(x, T, U, V, M, Nframes, FigNumber)
%
% 18.311 (Principles of Applied Mathematics). MIT, Winter 2008. R. R. Rosales.
% ____________________________________________________________________________
%
% Makes a movie with the output of InitGBNS,  showing the solution of the wave
% equation (with periodic boundary conditions) for:
% --- Two types of intial values  (a gaussian or a cosine),  parametrized by a
%      scalar a > 0 (see help for InitGBNS.)
% --- Two possible numerical schemes: "good" and "bad", see help for InitGBNS.
%
% Has TWO MODES of operation:
%
% --- DEMO MODE: call the script with no inputs or outputs.
%                A control figure is created, to let the user run the demo by:
%                a) Selecting parameters for initial values, scheme and movie.
%                b) Make the movie.
%                c) See the movie.
%                d) Get help.
%     In this mode the movie (and parameters)  can be recovered via the global
%     variable global_MovieGBNS_choose (a structure.) In particular, the movie
%     is stored in global_MovieGBNS_choose.Mv.  For programming details on how
%     the script  works in the demo mode, and how the global variable is used,
%     what fields it has, etc., see the help file: movieGBNS_NOTES.m
%
% --- MANUAL MODE: call the script with the inputs x, T, U, V, M, Nframes, and
%                  the (optional) input FigNumber.  The output Mv is optional.
%                  Then: * A movie with the input is created.
%                        * FigNumber -- see below -- controls which figure the
%                          movie is created on.
%                        * The option to run it once is given.
%                        * The movie is saved in the output Mv (if required.)
%
%     With the output Mv, the movie can be seen again using the commands:
%     --- movie(Mv, n) ........ Shows the movie n times.
%     --- movie(Mv, n, FPS) ... Shows the movie n times at FPS frames per sec.
%
%     INPUT/OUTPUTS for manual mode:
%
%     x, T, U, V, and M ... are exactly as in the output for InitGBNS.
%     Nframes ............. is the number of frames desired in the movie.  The
%                           The actual number will differ from this because of
%                           the need to make the frames equally spaced in time
%                           -- and the fact that the solution is known only at
%                           the M provided times.    The scheme will try to do
%                           the best it can for the M given.
%     FigNumber ........... A figure can be provided, where the movie is to be
%                           made.  This is done with this (optional) argument,
%                           which should be the  figure number for the desired
%                           figure. If none is provided, the script creates an
%                           appropriate one.
%
%     Mv .................. Array with movie (optional output.)
%
%     WARNING: Movies can use large amounts of memory.  If the figure provided
%              is too large, MatLab may run out of memory while making a movie
%     (or slowdown to a crawl because it starts using HD virtual memory.) Thus
%     it is important to use a SMALL figure window to make the movie.  Exactly
%     what "small" means depends on:
%     --- How much RAM is available.
%     --- How many frames are requested.
%     The following formula gives a  reasonably precise answer to the question
%     of how much memory is needed (each pixel takes a bit of info):
%
%         MegaBytes = PA*Nframes/(8*1024^2)
%
%     where  PA = Area of the plotting area in the figure (in PIXELS.)
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if nargin < 7; FigNumber = 0; end
%
% --- Loop for control from figure buttons.
%
if nargin == 0
   MovieGNBS_choose
   return
elseif nargin == 1
   MovieGNBS_choose(x)
   return
end
%
% --- Create figure if needed;  or bring to the
%     foreground the designated figure if given.
%
if abs(FigNumber) > 0.5;
   figure(abs(FigNumber))
else
   % --- Movie figure width = height. %
         SSZ = get(0, 'ScreenSize');  %
         if SSZ(3) > SSZ(4)           %
            FigD = 0.4*SSZ(4);        %
         else                         %
            FigD = 0.4*SSZ(3);        %
         end                          %
   % -------------------------------- %
   Position = [SSZ(3) SSZ(4) 0 0]+FigD*[-1.1 -1.2 1.0 1.0];
   FigNumb = figure( ...
              'NumberTitle', 'off', ...
              'Units',       'pixels', ...
              'Position',     Position, ...
              'Name',        'movie GNBS.');
   set(gca,   'Position',     [0.17 0.11 0.73 0.73])
   whitebg(FigNumb, 'k')
end
%
% --- Message to user when user provides the figure (NON DEMO mode.)
%
if FigNumber > 0.5
   fprintf('\n')
   disp( ...
     ['  ____________________________________________________________'; ...
      '  Movies can use large amounts of memory.    If you run out of'; ...
      '  memory while making the movie, try again but reduce the size'; ...
      '  of the window you provided for the movie.                   '; ...
      '  -------- PRESS ANY KEY TO START MAKING THE MOVIE. --------- '; ...
      '  ____________________________________________________________'])
   fprintf('\n')
   pause
end
%
% --- Figure out the  "skip"  needed to
%     get the desired number of frames.
%
Nframes = round(Nframes);     if Nframes < 1; Nframes = 1; end
%
if Nframes > 1
   nskip = max(1, floor((M-1)/(Nframes-1)));
else
   nskip = M + 1;
end
%
NFRAMES = length(1:nskip:M);
%
% --- Get movie done.
%
Mv   = moviein(NFRAMES);
RES  = num2str(length(x));
Umax = max(max(U));
Umin = min(min(U));
for n=1:NFRAMES
   nn = (n-1)*nskip + 1;
   cla
   set(gca, 'FontWeight', 'Bold', ...
            'FontUnits',  'Normalized', ...
            'FontSize',    0.07, ...
            'LineWidth',   1)
   hold on
   grid on
   plot(x, U(nn,:), '-c', 'LineWidth', 3)
   axis([-1 1 Umin Umax])
   ylabel('{\bf u }', 'FontUnits',           'Normalized', ...
                      'FontSize',             0.08, ...
                      'Rotation',             0, ...
                      'HorizontalAlignment', 'right')
   xlabel('{\bf x}',  'FontUnits',           'Normalized', ...
                      'FontSize',             0.08)
   text('String',              '{\bf Wave equation solution u=u(x,t).}', ...
        'FontUnits',           'Normalized', ...
        'FontSize',             0.075, ...
        'HorizontalAlignment', 'center', ...
        'Units' ,              'Normalized', ...
        'Position',             [0.45, 1.15])
   TIME = num2str(T(nn), '%4.2f');
   text('String',     ['{\bf t = ', TIME, '}'], ...
        'FontUnits', 'Normalized', ...
        'FontSize',   0.08, ...
        'Units' ,    'Normalized', ...
        'Position',   [0.65 0.90])
   text('String',             ['{\bf Points in space grid = ', RES, '.}'], ...
        'VerticalAlignment', 'bottom', ...
        'Rotation',           90, ...
        'FontUnits',         'Normalized', ...
        'FontSize',           0.08, ...
        'Units' ,            'Normalized', ...
        'Position',           [1.15, -0.02])

   hold off
   Mv(:,n) = getframe;
end
%
% --- Message to user in NON-DEMO mode.
%
if FigNumber >= 0
   fprintf('\n Press any key to see the movie play once. \n')
   pause
   movie(Mv)
   fprintf('\n If the output from this script is stored in Mv:')
   fprintf('\n --- movie(Mv, n) ......... shows the movie n times.')
   fprintf('\n --- movie(Mv, n, FPS) .... shows the movie n times,')
   fprintf('\n                            at FPS frames per second.\n')
end
%
% --- Clear Mv if no output requested.
%
if nargout == 0; clear Mv; end
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function MovieGNBS_choose(n)
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% function MovieGNBS_choose(n)
%
% See movieGBNS_NOTES.m for operational details.
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
global global_MovieGBNS_choose
%
% ____________________________________________________________________________
%
if nargin == 0
   %
   % a01 --- Create command figure and write title.
   %
   if isfield(global_MovieGBNS_choose, 'fcons')
      f = global_MovieGBNS_choose.fcons;
      if 1 == Exist_Figure_GBNS(f)
         delete(f)
      end
   end
   COL = [0.60 0.60 0.60];
   % --- Control figure width and height. --------- %
         SSZ = get(0, 'ScreenSize');                %
         if 3*SSZ(3) > 4*SSZ(4)                     %
            FigH = 0.5*SSZ(4);   FigW = (4/3)*FigH; %
         else                                       %
            FigW = 0.5*SSZ(3);   FigH = (3/4)*FigW; %
         end                                        %
   % ---------------------------------------------- %
   f = figure('Color',        COL, ...
              'NumberTitle', 'off', ...
              'Units',       'pixels', ...
              'Position',     [SSZ(3)-1.1*FigW, 0.133*FigH, FigW, FigH], ...
              'Name',        'Control figure for movieGNBS.');
   global_MovieGBNS_choose.fcons = f;
   axis off
   text('String',              'Choose parameter values for movie:', ...
        'Units',               'Normalized', ...
        'FontUnits',           'Normalized', ...
        'FontWeight',          'Bold', ...
        'HorizontalAlignment', 'Center', ...
        'FontSize',             0.08, ...
        'Position',             [0.50, 1.00])
   %
   % a02 --- Create (exclusive) radio buttons (2) to select initial values.
   %
   val = 1;
   if isfield(global_MovieGBNS_choose, 'INIVAL')
      INIVAL = global_MovieGBNS_choose.INIVAL;
      if strcmp(INIVAL, 'C')
         val = 0;
      end
   else
      global_MovieGBNS_choose.INIVAL = 'G';
   end
   STRING = 'Initial Value: u =10^(-a*x^2); gaussian.';
   h = uicontrol('Style',           'radiobutton', ...
                 'FontUnits',       'Normalized', ...
                 'FontSize',         0.50, ...
                 'Value',            val, ...
                 'FontWeight',      'bold', ...
                 'BackGroundColor',  COL, ...
                 'Units',           'normalized', ...
                 'Position',         [0.05 0.81 0.90 0.05], ...
                 'Callback',        'movieGBNS(1)', ...
                 'String',           STRING);
   global_MovieGBNS_choose.h1 = h;
   %
   STRING = 'Initial Value: u = 0.5*(1+cos(a*pi*x)); cosine.';
   h = uicontrol('Style',           'radiobutton', ...
                 'FontUnits',       'Normalized', ...
                 'FontSize',         0.50, ...
                 'Value',            1-val, ...
                 'FontWeight',      'bold', ...
                 'BackGroundColor',  COL, ...
                 'Units',           'normalized', ...
                 'Position',         [0.05 0.75 0.90 0.05], ...
                 'Callback',        'movieGBNS(2)', ...
                 'String',           STRING);
   global_MovieGBNS_choose.h2 = h;
   %
elseif n == 1
   %
   % b01 --- Select Initial Value = Gaussian.
   %
   val = get(global_MovieGBNS_choose.h1, 'Value');
   set(global_MovieGBNS_choose.h2, 'Value', 1-val)
   if val == 1
      global_MovieGBNS_choose.INIVAL = 'G';
   else
      global_MovieGBNS_choose.INIVAL = 'C';
   end
   %
elseif n == 2
   %
   % b02 --- Select Initial Value = Cosine.
   %
   val = get(global_MovieGBNS_choose.h2, 'Value');
   set(global_MovieGBNS_choose.h1, 'Value', 1-val)
   if val == 1
      global_MovieGBNS_choose.INIVAL = 'C';
   else
      global_MovieGBNS_choose.INIVAL = 'G';
   end
end
%
% ____________________________________________________________________________
%
if nargin == 0
   %
   % a03 --- Create edit window to select value of a.
   %
   Nstring = '1.000';
   if isfield(global_MovieGBNS_choose, 'a')
      Nstring = num2str(global_MovieGBNS_choose.a);
   else
      global_MovieGBNS_choose.a = 1.000;
   end
   h = uicontrol('Style',           'Edit', ...
                 'FontUnits',       'Normalized', ...
                 'FontSize',         0.45, ...
                 'FontWeight',      'bold', ...
                 'BackGroundColor', 'w', ...
                 'ForeGroundColor', 'r', ...
                 'Units',           'normalized', ...
                 'Position',         [0.05 0.66 0.30 0.07], ...
                 'Callback',        'movieGBNS(3)', ...
                 'String',           Nstring);
   global_MovieGBNS_choose.h3 = h;
   uicontrol('Style',               'Text', ...
             'FontUnits',           'Normalized', ...
             'FontSize',             0.50, ...
             'FontWeight',          'Bold', ...
             'BackGroundColor',      COL, ...
             'ForeGroundColor',     'k', ...
             'Units',               'Normalized', ...
             'String',              'Select value for a > 0.', ...
             'HorizontalAlignment', 'Left', ...
             'Position',             [0.36, 0.67, 0.50, 0.05])
elseif n == 3
   %
   % b03 --- Select value of a.
   %
   Nstring = get(global_MovieGBNS_choose.h3, 'String');
   a = abs(str2num(Nstring));
   if isempty(a)
      a = 1.000;
   elseif a < eps
      a = eps;
   end
   set(global_MovieGBNS_choose.h3, 'String', num2str(a, '%11.5e'))
   global_MovieGBNS_choose.a = a;
end
%
% ____________________________________________________________________________
%
if nargin == 0
   %
   % a04 --- Create (exclusive) radio buttons (2) to select SCHEME.
   %
   val = 1;
   if isfield(global_MovieGBNS_choose, 'SCHEME')
      SCHEME = global_MovieGBNS_choose.SCHEME;
      if strcmp(SCHEME, 'B')
         val = 0;
      end
   else
      global_MovieGBNS_choose.SCHEME = 'G';
   end
   STRING = ' Select scheme: "GOOD".';
   h = uicontrol('Style',           'radiobutton', ...
                 'FontUnits',       'Normalized', ...
                 'FontSize',         0.50, ...
                 'Value',            val, ...
                 'FontWeight',      'bold', ...
                 'BackGroundColor',  COL, ...
                 'Units',           'normalized', ...
                 'Position',         [0.05 0.59 0.50 0.05], ...
                 'Callback',        'movieGBNS(4)', ...
                 'String',           STRING);
   global_MovieGBNS_choose.h4 = h;
   %
   STRING =  ' Select scheme: "BAD".';
   h = uicontrol('Style',           'radiobutton', ...
                 'FontUnits',       'Normalized', ...
                 'FontSize',         0.50, ...
                 'Value',            1-val, ...
                 'FontWeight',      'bold', ...
                 'BackGroundColor',  COL, ...
                 'Units',           'normalized', ...
                 'Position',         [0.05 0.53 0.50 0.05], ...
                 'Callback',        'movieGBNS(5)', ...
                 'String',           STRING);
   global_MovieGBNS_choose.h5 = h;
   %
elseif n == 4
   %
   % b04 --- Select good scheme.
   %
   val = get(global_MovieGBNS_choose.h4, 'Value');
   set(global_MovieGBNS_choose.h5, 'Value', 1-val)
   if val == 1
      global_MovieGBNS_choose.SCHEME = 'G';
   else
      global_MovieGBNS_choose.SCHEME = 'B';
   end
   %
elseif n == 5
   %
   % b05 --- Select bad scheme.
   %
   val = get(global_MovieGBNS_choose.h5, 'Value');
   set(global_MovieGBNS_choose.h4, 'Value', 1-val)
   if val == 1
      global_MovieGBNS_choose.SCHEME = 'B';
   else
      global_MovieGBNS_choose.SCHEME = 'G';
   end
end
%
% ____________________________________________________________________________
%
if nargin == 0
   %
   % a05 --- Create edit window to select value of t_f.
   %
   Nstring = '2.000';
   if isfield(global_MovieGBNS_choose, 't_f')
      Nstring = num2str(global_MovieGBNS_choose.t_f);
   else
      global_MovieGBNS_choose.t_f = 2.000;
   end
   h = uicontrol('Style',           'Edit', ...
                 'FontUnits',       'Normalized', ...
                 'FontSize',         0.45, ...
                 'FontWeight',      'bold', ...
                 'BackGroundColor', 'w', ...
                 'ForeGroundColor', 'r', ...
                 'Units',           'normalized', ...
                 'Position',         [0.05 0.43 0.30 0.07], ...
                 'Callback',        'movieGBNS(6)', ...
                 'String',           Nstring);
   global_MovieGBNS_choose.h6 = h;
   STRING = 'Select computation time t_f > 0.';
   uicontrol('Style',               'Text', ...
             'FontUnits',           'Normalized', ...
             'FontSize',             0.50, ...
             'FontWeight',          'Bold', ...
             'BackGroundColor',      COL, ...
             'ForeGroundColor',     'k', ...
             'Units',               'Normalized', ...
             'String',               STRING, ...
             'HorizontalAlignment', 'Left', ...
             'Position',             [0.36, 0.44, 0.60, 0.05])
elseif n == 6
   %
   % b06 --- Enter value of t_f in edit window.
   %
   Nstring = get(global_MovieGBNS_choose.h6, 'String');
   t_f = abs(str2num(Nstring));
   if isempty(t_f)
      t_f = 2.000;
   elseif t_f < eps
      t_f = eps;
   end
   set(global_MovieGBNS_choose.h6, 'String', num2str(t_f, '%11.5e'))
   global_MovieGBNS_choose.t_f = t_f;
end
%
% ____________________________________________________________________________
%
if nargin == 0
   %
   % a06 --- Create edit window to select value of N.
   %
   Nstring = '50';
   if isfield(global_MovieGBNS_choose, 'N')
      Nstring = num2str(global_MovieGBNS_choose.N);
   else
      global_MovieGBNS_choose.N = 50;
   end
   h = uicontrol('Style',           'Edit', ...
                 'FontUnits',       'Normalized', ...
                 'FontSize',         0.45, ...
                 'FontWeight',      'bold', ...
                 'BackGroundColor', 'w', ...
                 'ForeGroundColor', 'r', ...
                 'Units',           'normalized', ...
                 'Position',         [0.05 0.35 0.15 0.07], ...
                 'Callback',        'movieGBNS(7)', ...
                 'String',           Nstring);
   global_MovieGBNS_choose.h7 = h;
   STRING = 'Select number of points in x-grid.';
   uicontrol('Style',               'Text', ...
             'FontUnits',           'Normalized', ...
             'FontSize',             0.50, ...
             'FontWeight',          'Bold', ...
             'BackGroundColor',      COL, ...
             'ForeGroundColor',     'k', ...
             'Units',               'Normalized', ...
             'String',               STRING, ...
             'HorizontalAlignment', 'Left', ...
             'Position',             [0.21, 0.36, 0.70, 0.05])
elseif n == 7
   %
   % b07 --- Enter value of N in edit window.
   %
   Nstring = get(global_MovieGBNS_choose.h7, 'String');
   N = round(abs(str2num(Nstring)));
   if isempty(N)
      N = 50;
   elseif N < 2
      N = 2;
   end
   set(global_MovieGBNS_choose.h7, 'String', num2str(N))
   global_MovieGBNS_choose.N = N;
end
%
% ____________________________________________________________________________
%
if nargin == 0
   %
   % a07 --- Create edit window to select value of Nframes.
   %
   Nstring = '25';
   if isfield(global_MovieGBNS_choose, 'Nframes')
      Nstring = num2str(global_MovieGBNS_choose.Nframes);
   else
      global_MovieGBNS_choose.Nframes = 25;
   end
   h = uicontrol('Style',           'Edit', ...
                 'FontUnits',       'Normalized', ...
                 'FontSize',         0.45, ...
                 'FontWeight',      'bold', ...
                 'BackGroundColor', 'w', ...
                 'ForeGroundColor', 'r', ...
                 'Units',           'normalized', ...
                 'Position',         [0.05 0.27 0.15 0.07], ...
                 'Callback',        'movieGBNS(8)', ...
                 'String',           Nstring);
   global_MovieGBNS_choose.h8 = h;
   STRING = 'Select number of frames in movie.';
   uicontrol('Style',               'Text', ...
             'FontUnits',           'Normalized', ...
             'FontSize',             0.50, ...
             'FontWeight',          'Bold', ...
             'BackGroundColor',      COL, ...
             'ForeGroundColor',     'k', ...
             'Units',               'Normalized', ...
             'String',               STRING, ...
             'HorizontalAlignment', 'Left', ...
             'Position',             [0.21, 0.28, 0.70, 0.05])
elseif n == 8
   %
   % b08 --- Enter value of Nframes in edit window.
   %
   Nstring = get(global_MovieGBNS_choose.h8, 'String');
   Nframes = round(abs(str2num(Nstring)));
   if isempty(Nframes)
      Nframes = 25;
   elseif Nframes < 2
      Nframes = 2;
   end
   set(global_MovieGBNS_choose.h8, 'String', num2str(Nframes))
   global_MovieGBNS_choose.Nframes = Nframes;
end
%
% ____________________________________________________________________________
%
if nargin == 0
   %
   % a08 --- Create edit window to select value of FPS.
   %
   Nstring = '12';
   if isfield(global_MovieGBNS_choose, 'FPS')
      Nstring = num2str(global_MovieGBNS_choose.FPS);
   else
      global_MovieGBNS_choose.FPS = 12;
   end
   h = uicontrol('Style',           'Edit', ...
                 'FontUnits',       'Normalized', ...
                 'FontSize',         0.45, ...
                 'FontWeight',      'bold', ...
                 'BackGroundColor', 'w', ...
                 'ForeGroundColor', 'r', ...
                 'Units',           'normalized', ...
                 'Position',         [0.05 0.19 0.15 0.07], ...
                 'Callback',        'movieGBNS(9)', ...
                 'String',           Nstring);
   global_MovieGBNS_choose.h9 = h;
   STRING = 'Select frames/second for movie.';
   uicontrol('Style',               'Text', ...
             'FontUnits',           'Normalized', ...
             'FontSize',             0.50, ...
             'FontWeight',          'Bold', ...
             'BackGroundColor',      COL, ...
             'ForeGroundColor',     'k', ...
             'Units',               'Normalized', ...
             'String',               STRING, ...
             'HorizontalAlignment', 'Left', ...
             'Position',             [0.21, 0.20, 0.70, 0.05])
elseif n == 9
   %
   % b09 --- Enter value of FPS in edit window.
   %
   Nstring = get(global_MovieGBNS_choose.h9, 'String');
   FPS = round(abs(str2num(Nstring)));
   if isempty(FPS)
      FPS = 12;
   elseif FPS < 1
      FPS = 1;
   end
   set(global_MovieGBNS_choose.h9, 'String', num2str(FPS))
   global_MovieGBNS_choose.FPS = FPS;
end
%
% ____________________________________________________________________________
%
if nargin == 0
   %
   % a09 --- Make movie button.
   %
   h = uicontrol('Style',           'Pushbutton', ...
                 'FontUnits',       'Normalized', ...
                 'FontSize',         0.45, ...
                 'FontWeight',      'bold', ...
                 'BackGroundColor', 'c', ...
                 'ForeGroundColor', 'b', ...
                 'Units',           'normalized', ...
                 'Position',         [0.05 0.01 0.35 0.12], ...
                 'Callback',        'movieGBNS(10)', ...
                 'String',          'Make movie.');
   global_MovieGBNS_choose.h10 = h;
   %
elseif n == 10
   %
   % b10 --- Make movie. If movie figure does not exist, create it.
   %
         N = global_MovieGBNS_choose.N;
         a = global_MovieGBNS_choose.a;
       t_f = global_MovieGBNS_choose.t_f;
    INIVAL = global_MovieGBNS_choose.INIVAL;
    SCHEME = global_MovieGBNS_choose.SCHEME;
   Nframes = global_MovieGBNS_choose.Nframes;
   %
   [x, T, U, V, M] = InitGBNS(N, a, t_f, INIVAL, SCHEME, Nframes);
   %
   MakeFig = 1;
   if isfield(global_MovieGBNS_choose, 'fmovie')
      f = global_MovieGBNS_choose.fmovie;
      if 1 == Exist_Figure_GBNS(f)
         MakeFig = 0;
      end
   end
   %
   if MakeFig == 1
      % --- Movie figure width = height. %
            SSZ = get(0, 'ScreenSize');  %
            if 3*SSZ(3) > 4*SSZ(4)       %
               FigD = 0.4*SSZ(4);        %
            else                         %
               FigD = 0.3*SSZ(3);        %
            end                          %
      % -------------------------------- %
      f = figure( ...
              'NumberTitle', 'off', ...
              'Units',       'pixels', ...
              'Position',     [0 SSZ(4) 0 0] + FigD*[0.1 -1.2 1.0 1.0], ...
              'Name',        'movie GNBS.');
      set(gca,   'Position',     [0.17 0.11 0.73 0.73])
      whitebg(f, 'k')
      global_MovieGBNS_choose.fmovie = f;
   end
   %
   Mv = movieGBNS(x, T, U, V, M, Nframes, -f);
   global_MovieGBNS_choose.Mv = Mv;
   set(global_MovieGBNS_choose.h11, 'Visible', 'on')
   figure(global_MovieGBNS_choose.fcons)
   %
   % --- Disable buttons and edit windows so the parameters
   %     cannot be changed accidentally.
   set(global_MovieGBNS_choose.h1, 'Enable', 'off')
   set(global_MovieGBNS_choose.h2, 'Enable', 'off')
   set(global_MovieGBNS_choose.h3, 'Enable', 'off')
   set(global_MovieGBNS_choose.h4, 'Enable', 'off')
   set(global_MovieGBNS_choose.h5, 'Enable', 'off')
   set(global_MovieGBNS_choose.h6, 'Enable', 'off')
   set(global_MovieGBNS_choose.h7, 'Enable', 'off')
   set(global_MovieGBNS_choose.h8, 'Enable', 'off')
   %
   % --- Switch "Make movie" button to: "New movie"
   %
   set(global_MovieGBNS_choose.h10, ...
       'String',          'New movie?', ...
       'CallBack',        'movieGBNS(10.5)', ...
       'ForeGroundColor', 'y', ...
       'BackGroundColor', 'r')
   %
elseif abs(n-10.5) < 0.01
   %
   % --- Get things back to allow another movie.
   %
   set(global_MovieGBNS_choose.h1, 'Enable', 'on')
   set(global_MovieGBNS_choose.h2, 'Enable', 'on')
   set(global_MovieGBNS_choose.h3, 'Enable', 'on')
   set(global_MovieGBNS_choose.h4, 'Enable', 'on')
   set(global_MovieGBNS_choose.h5, 'Enable', 'on')
   set(global_MovieGBNS_choose.h6, 'Enable', 'on')
   set(global_MovieGBNS_choose.h7, 'Enable', 'on')
   set(global_MovieGBNS_choose.h8, 'Enable', 'on')
   %
   set(global_MovieGBNS_choose.h10, ...
       'String',          'Make movie.', ...
       'CallBack',        'movieGBNS(10)', ...
       'ForeGroundColor', 'b', ...
       'BackGroundColor', 'c')
   set(global_MovieGBNS_choose.h11, 'Visible', 'off')
end
%
% ____________________________________________________________________________
%
if nargin == 0
   %
   % a10 --- Make Play movie button.
   %         Make visible only if there is a movie to play.
   %
   h = uicontrol('Style',           'Pushbutton', ...
                 'FontUnits',       'Normalized', ...
                 'FontSize',         0.45, ...
                 'FontWeight',      'bold', ...
                 'BackGroundColor', 'y', ...
                 'ForeGroundColor', 'b', ...
                 'Units',           'normalized', ...
                 'Position',         [0.50 0.01 0.35 0.12], ...
                 'Callback',        'movieGBNS(11)', ...
                 'Visible',         'off', ...
                 'String',          'Play movie.');
   global_MovieGBNS_choose.h11 = h;
%    if isfield(global_MovieGBNS_choose, 'Mv')
%       if ~isempty(global_MovieGBNS_choose.Mv)
%          set(h, 'Visible', 'on')
%       end
%    end
elseif n == 11
   %
   % b11 --- Play movie.
   %
   figure(global_MovieGBNS_choose.fmovie)
   FPS = global_MovieGBNS_choose.FPS;
   movie(global_MovieGBNS_choose.Mv, 1, FPS)
   figure(global_MovieGBNS_choose.fcons)
end
%
% ____________________________________________________________________________
%
if nargin == 0
   %
   % a11 --- Make help button.
   %
   h = uicontrol('Style',           'Pushbutton', ...
                 'FontUnits',       'Normalized', ...
                 'FontSize',         0.45, ...
                 'FontWeight',      'bold', ...
                 'BackGroundColor', 'g', ...
                 'ForeGroundColor', 'k', ...
                 'Units',           'normalized', ...
                 'Position',         [0.55 0.51 0.20 0.14], ...
                 'Callback',        'movieGBNS(12)', ...
                 'String',          'Help.');
   global_MovieGBNS_choose.h12 = h;
   %
elseif n == 12
   %
   % b12 --- Open help window.
   %
   HelpTxT = [ ...
   ' This script makes a movie for the solution of the wave equation'; ...
   '                       u_tt - u_xx = 0,                         '; ...
   ' for 0 <= t <= t_f                                              '; ...
   ' and periodic boundary conditions for -1 <= x <= 1.             '; ...
   ' Two choices of initial conditions are allowed,  and two choices'; ...
   ' of numerical schemes, given as follows:                        '; ...
   '                                                                '; ...
   ' Let v = u_t (the equation is then u_t = v and v_t = u_xx.)     '; ...
   ' Let dt and dx be the time and space steps (respectively) in the'; ...
   ' numerical approximation. Then:                                 '; ...
   '                                                                '; ...
   ' SCHEME B (B for Bad):      u(x,t+dt) = u(x,t) +  v(x,t)*dt,    '; ...
   '                            v(x,t+dt) = v(x,t) + u2(x,t)*nu,    '; ...
   '                                                                '; ...
   ' SCHEME G (G for Good):                                         '; ...
   '                u(x,t+dt) = u(x,t) +  v(x,t)*dt + u2(x,t)*eta,  '; ...
   '                v(x,t+dt) = v(x,t) + u2(x,t)*nu + v2(x,t)*eta,  '; ...
   '                                                                '; ...
   ' where nu  = dt/(dx)^2,                                         '; ...
   '       eta = (dt/dx)^2,                                         '; ...
   '       u2 = u(x+dx, t) - 2 u(x, t) + u(x-dx, t),                '; ...
   ' and   v2 = v(x+dx, t) - 2 v(x, t) + v(x-dx, t).                '; ...
   '                                                                '; ...
   ' The (naive) scheme B is unstable. Scheme G has been made stable'; ...
   ' by adding "numerical" dissipation.                             '; ...
   '                                                                '; ...
   ' The purpose of this script is to illustrate how bad a numerical'; ...
   ' approximation can be, and why care must be taken when designing'; ...
   ' a scheme for computation.                                      '; ...
   '                                                                '];
   helpwin(HelpTxT, 'Help for movieGBNS')
end
%
% ____________________________________________________________________________
%
if nargin == 0
   %
   % a12 --- Make Quit button.
   %
   h = uicontrol('Style',           'Pushbutton', ...
                 'FontUnits',       'Normalized', ...
                 'FontSize',         0.45, ...
                 'FontWeight',      'bold', ...
                 'BackGroundColor', 'y', ...
                 'ForeGroundColor', 'r', ...
                 'Units',           'normalized', ...
                 'Position',         [0.77 0.51 0.20 0.14], ...
                 'Callback',        'movieGBNS(13)', ...
                 'String',          'Quit.');
   global_MovieGBNS_choose.h13 = h;
   %
elseif n == 13
   %
   % b13 --- Close windows and delete global variable.
   %
   fcons  = global_MovieGBNS_choose.fcons;
   fmovie = 0.5;
   if isfield(global_MovieGBNS_choose, 'fmovie')
      fmovie = global_MovieGBNS_choose.fmovie;
   end
   if Exist_Figure_GBNS(fcons)
      close(fcons)
   end
   %
   if Exist_Figure_GBNS(fmovie)
      close(fmovie)
   end
   %
   clear global global_MovieGBNS_choose
end
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function flag = Exist_Figure_GBNS(Fhandle)
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% function flag = Exist_Figure_GBNS(Fhandle)
%
% flag = 1 if figure with handle Fhandle is     currently displayed on screen.
% flag = 0 if """"""""""""""""""""""""""""" NOT """""""""""""""""""""""""""""
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
fighdl = [];
hdlchi = get(0, 'Children');
for n = 1:length(hdlchi)
   if strcmp(get(hdlchi(n), 'type'), 'figure')
      fighdl = [fighdl; hdlchi(n)];
   end
end
fighdl = sort(fighdl);
flag = 0;
if ~isempty(fighdl)
   flag = max(fighdl == Fhandle);
end
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Copyright 2001-2008 Massachusetts Institute of Technology
% Version 01 by Rodolfo R. Rosales  02-15-2001.
% Update     by Rodolfo R. Rosales  02-25-2008.
% 
% Permission is hereby granted, without payment, to copy this software
% and its documentation, if any,  for non-profit academic and research
% purposes,  provided that the above copyright notice, this paragraph,
% and the following three paragraphs appear in all copies of this
% software. Use of this software constitutes acceptance of these terms
% and conditions.
%
% IN NO EVENT SHALL MIT, OR THE AUTHOR, BE LIABLE TO ANY PARTY FOR
% DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
% ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
% IF MIT, OR THE AUTHOR, HAVE BEEN ADVISED OF THE POSSIBILITY OF SUCH
% DAMAGE.
%
% MIT, AND THE AUTHOR, SPECIFICALLY DISCLAIM ANY EXPRESS OR IMPLIED
% WARRANTIES INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
% MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, AND
% NON-INFRINGEMENT.
%
% THIS SOFTWARE IS PROVIDED "AS IS." MIT, OR THE AUTHOR, HAVE NO
% OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS,
% OR MODIFICATIONS.
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% EOF
