function OUTPUT = demoWBRch_v03(EXAMPLE, N)
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% function [Fig3D, FigSL, FigHW] = demoWBRch_v03(EXAMPLE)
% ____________________________________________________________________________
% 18.311 (Principles of Applied Mathematics). MIT, Winter 2008. R. R. Rosales.
%
% This demo shows various examples of the time evolution given by the equation
%
%                        u_t + c(u)*u_x = 0.
% 
% In all cases the evolution from initial conditions given for x in [0,  2*pi]
% are shown. The solution is calculated using CHARACTERISTICS.
%                ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
% See also evolveWBRch, which gives more flexibility.
%
% INPUT/OUTPUT ===============================================================
% EXAMPLE ... Optional. Integer with the example number (see list below) to be
%             displayed. If not provided script starts with EXAMPLE = 1.  Then
%             buttons allow the user to in cycles down the list.
%
%             (1) c=u       and u_0 = sin(x).
%             (2) c=u       and u_0 = exp(-(x-pi)^2).
%             (3) c=(1-2*u) and u_0 = 0.5*(1-sin(x)).
%             (4) c=(1-2*u) and u_0 = 0.5*(1-sin(x))+0.6*exp(-3*(x-2)^2).
%             (5) c=(1-2*u) and u_0 = Zig-Zag.
%             (6) c=u       and u_0 = "wavy".
%             (7) c=sqrt(u) and u_0 = 0.2 + 0.7*exp(-x^2).
%
% SPECIAL USAGE: The single input (when being called by the user, the other
%                input N is only used by the CallBacks from buttons --- see
%                demoWBRch_v03_HowWork) EXAMPLE can also be a cell array:
%
%                EXAMPLE{1} = Integer with the selected example number.
%                EXAMPLE{2} = String with MatLab commands,  to be executed
%                             when the script is terminated --- by pushing
%                             one of the close buttons.
%
% OUTPUT (optional) is a structure with the fields listed below. Each field
% contains the handle for the object listed next to the field name:
%
% Fig3D ......... Figure with the 3D plot.
% Fig3Dexmpl .... """"""""""""""""""""""": Example button.
% Fig3Dclose .... """"""""""""""""""""""": Close   button.
% FigSL ......... Figure with the slider.
% FigSLslidr .... """""""""""""""""""""" : Slider.
% FigSLmovie .... """""""""""""""""""""" : Movie/Stop button.
% FigSLexmpl .... """""""""""""""""""""" : Example button.
% FigSLclose .... """""""""""""""""""""" : Close   button.
% FigHW ......... Help Window figure.
% FigHWexmpl .... """""""""""""""""":      List Box.
% FigHWclose .... """""""""""""""""":      Close button.
% FigHWpopup .... """""""""""""""""":      Popup menu.
%
% WARNING: if a figure window is closed during operation,  when using the
%          example advance button it will be recreated,   however it (and
%          its children)  will have different handles from the ones given
%          by OUTPUT.
%
% CHANGES RELATIVE TO PRIOR VERSIONS =========================================
% version 00 demoWBRch
% --- Show is controlled entirely from the keyboard.
% --- Cannot go back in time evolution. One shot deal.
% version 01 demoWBRch_v01
% --- Added slider to control which plot is shown.
% --- Improved looks of the figure windows (better labels, etc).
% --- Can skip the options menu by directly calling a choice via an input.
% version 02 demoWBRch_v02
% --- Added output with handles for figures.
% --- Added control button to switch between movie show and slider show.
% --- Added example button to switch between examples.
% --- Added close   button to stop demo.
% --- Added help window.
% version 03 demoWBRch_v03
% --- Changed the way internal operations are carried, to avoid problems when
%     windows are accidentally closed, and such.
% --- Added an example control button to the 3D figure as well.
% --- Added as an output the handles for all the objects.
% --- Added popup menu in Help Window.
% --- Added option to have a user selected command when the script is
%     terminated by a close button.
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Copyright 1999-2008 Massachusetts Institute of Technology
% Version 00 by Rodolfo R. Rosales  02-25-1999.
% Version 01 by Rodolfo R. Rosales  03-24-1999.
% Version 02 by Rodolfo R. Rosales  03-27-2002.
% Version 03 by Rodolfo R. Rosales  03-30-2002.
% Update     by Rodolfo R. Rosales  03-10-2008.
% 
% Permission is hereby granted, without payment, to copy this software
% and its documentation, if any,  for non-profit academic and research
% purposes,  provided that the above copyright notice, this paragraph,
% and the following three paragraphs appear in all copies of this
% software. Use of this software constitutes acceptance of these terms
% and conditions.
%
% IN NO EVENT SHALL MIT, OR THE AUTHOR, BE LIABLE TO ANY PARTY FOR
% DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
% ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
% IF MIT, OR THE AUTHOR, HAVE BEEN ADVISED OF THE POSSIBILITY OF SUCH
% DAMAGE.
%
% MIT, AND THE AUTHOR, SPECIFICALLY DISCLAIM ANY EXPRESS OR IMPLIED
% WARRANTIES INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
% MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, AND
% NON-INFRINGEMENT.
%
% THIS SOFTWARE IS PROVIDED "AS IS." MIT, OR THE AUTHOR, HAVE NO
% OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS,
% OR MODIFICATIONS.
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if nargin == 0
   EXAMPLE = 1;
end
% ============================================================================
% How the script works: see demoWBRch_v03_HowWork
% ============================================================================
if nargin <= 1
   %
   CloseComnd = '';
   if iscell(EXAMPLE)
      CloseComnd = EXAMPLE{2};
      EXAMPLE    = EXAMPLE{1};
   end
   %
   % --- Calculate Example.
   %
   Output = demoWBRch_v03_Example(EXAMPLE);
   OutAux = Output;
   if isempty(OutAux)
      EXAMPLE = 1;
      Output = demoWBRch_v03_Example(EXAMPLE);
   end
   %
   % --- Make Fig3D and do 3D plot.
   %
   [Fig3D, Fig3Dexmpl, Fig3Dclose] = demoWBRch_v03_Fig3D(Output);
   %
   % --- Make FigSL and do startup screen.
   %
   [FigSL, FigSLslidr, FigSLmovie, FigSLexmpl, FigSLclose] = ...
      demoWBRch_v03_FigSL(Output);
   %
   % --- Make FigHW
   %
   [FigHW, FigHWexmpl, FigHWclose, FigHWpopup] = ...
      demoWBRch_v03_FigHW(Output.Lexa);
   %
   % --- SetUp the UserData structures.
   %
   UserData = Output;
   UserData.CloseComnd = CloseComnd;
   UserData.Fig3D      = Fig3D;
   UserData.Fig3Dexmpl = Fig3Dexmpl;
   UserData.Fig3Dclose = Fig3Dclose;
   UserData.FigSL      = FigSL;
   UserData.FigSLslidr = FigSLslidr;
   UserData.FigSLmovie = FigSLmovie;
   UserData.FigSLexmpl = FigSLexmpl;
   UserData.FigSLclose = FigSLclose;
   UserData.FigHW      = FigHW;
   UserData.FigHWexmpl = FigHWexmpl;
   UserData.FigHWclose = FigHWclose;
   UserData.FigHWpopup = FigHWpopup;
   %
   set(Fig3Dexmpl, 'UserData', UserData)
   set(FigSLexmpl, 'UserData', UserData)
   set(FigHWexmpl, 'UserData', UserData)
   %
   % --- SetUp the callbacks for the buttons.
   %
   Handles = [FigSLexmpl, Fig3Dexmpl, FigHWexmpl];
   HdlsTxt = ['[', num2str(Handles, '%37.30e'), ']'];
   set(FigSLslidr, 'CallBack', ['demoWBRch_v03(', HdlsTxt, ', 0)'])
   set(FigSLmovie, 'CallBack', ['demoWBRch_v03(', HdlsTxt, ', 1)'])
   set(FigSLexmpl, 'CallBack', ['demoWBRch_v03(', HdlsTxt, ', 3)'])
   set(FigSLclose, 'CallBack', ['demoWBRch_v03(', HdlsTxt, ', 4)'])
   set(Fig3Dexmpl, 'CallBack', ['demoWBRch_v03(', HdlsTxt, ', 5)'])
   set(Fig3Dclose, 'CallBack', ['demoWBRch_v03(', HdlsTxt, ', 6)'])
   set(FigHWclose, 'CallBack', ['demoWBRch_v03(', HdlsTxt, ', 7)'])
   set(FigHWpopup, 'CallBack', ['demoWBRch_v03(', HdlsTxt, ', 8)'])
   %
   % --- SetUp string and value for the popupmenu.
   %
   Nexa = Output.Nexa;
   STRG = 'Example # 01. ';
   for n=2:Nexa
      STRG = [STRG, '|Example #', num2str(n, '%02.0f'), '. '];
   end
   set(FigHWpopup, 'String', STRG, 'Value', EXAMPLE)
   %
   if nargout == 1
      OUTPUT.Fig3D      = Fig3D;
      OUTPUT.Fig3Dexmpl = Fig3Dexmpl;
      OUTPUT.Fig3Dclose = Fig3Dclose;
      OUTPUT.FigSL      = FigSL;
      OUTPUT.FigSLslidr = FigSLslidr;
      OUTPUT.FigSLmovie = FigSLmovie;
      OUTPUT.FigSLexmpl = FigSLexmpl;
      OUTPUT.FigSLclose = FigSLclose;
      OUTPUT.FigHW      = FigHW;
      OUTPUT.FigHWexmpl = FigHWexmpl;
      OUTPUT.FigHWclose = FigHWclose;
      OUTPUT.FigHWpopup = FigHWpopup;
   end
   drawnow
   return
end
%
% --- Stuff below will execute only if nargin > 1, so N is defined. ==========
%
if N == 0;% ------- Slider call. ==================================
          %
   demoWBRch_v03_Slider(EXAMPLE(1))
   drawnow
              %
elseif N == 1;% --- Movie button call. ============================
              %
   demoWBRch_v03_movie(EXAMPLE(1))
              %
elseif N == 2;% --- Stop button call. =============================
              %
   UD = get(EXAMPLE(1), 'UserData');
   %
   % --- Reset Stop button to Movie button.
   %
   Handles = [UD.FigSLexmpl, UD.Fig3Dexmpl, UD.FigHWexmpl];
   HdlsTxt = ['[', num2str(Handles, '%37.30e'), ']'];
   CallBck = ['demoWBRch_v03(', HdlsTxt, ', 1)'];
   set(UD.FigSLmovie, ...
      'String',           'Movie', ...
      'BackGroundColor',  'c',    ...
      'UserData',         0, ...
      'CallBack',         CallBck)
              %
elseif N == 3;% --- Example button call (FigSL). ==================
              %
   demoWBRch_v03_ExFigSL(EXAMPLE(1))
              %
elseif N == 4;% --- Close   button call (FigSL). ==================
              %
   UserData = get(EXAMPLE(1), 'UserData');
   if ishandle(UserData.FigSL); close(UserData.FigSL); end
   if ishandle(UserData.Fig3D); close(UserData.Fig3D); end
   if ishandle(UserData.FigHW); close(UserData.FigHW); end
   eval([UserData.CloseComnd, ';'])
              %
elseif N == 5;% --- Example button call (Fig3D). ==================
              %
   demoWBRch_v03_ExFig3D(EXAMPLE(2))
              %
elseif N == 6;% --- Close   button call (Fig3D). ==================
              %
   UserData = get(EXAMPLE(2), 'UserData');
   if ishandle(UserData.FigSL); close(UserData.FigSL); end
   if ishandle(UserData.Fig3D); close(UserData.Fig3D); end
   if ishandle(UserData.FigHW); close(UserData.FigHW); end
   eval([UserData.CloseComnd, ';'])
              %
elseif N == 7;% --- Close   button call (FigHW). ==================
              %
   UserData = get(EXAMPLE(3), 'UserData');
   if ishandle(UserData.FigSL); close(UserData.FigSL); end
   if ishandle(UserData.Fig3D); close(UserData.Fig3D); end
   if ishandle(UserData.FigHW); close(UserData.FigHW); end
   eval([UserData.CloseComnd, ';'])
              %
elseif N == 8;% --- Popup menu call (FigHW). ======================
              %
   demoWBRch_v03_popup(EXAMPLE(3))
end
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%% END OF FILE
