function [Fig3D, FigSL] = demoWBRch_v02(OPTION, N)
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% function demoWBRch_v02(OPTION)
% ____________________________________________________________________________
% 18.311 (Principles of Applied Mathematics). MIT, Winter 2008. R. R. Rosales.
%
% This demo shows various examples of the time evolution given by the equation
%
%                        u_t + c(u)*u_x = 0.
% 
% In all cases the evolution from initial conditions given for x in [0,  2*pi]
% are shown. The solution is calculated using CHARACTERISTICS
%
% See also evolveWBRch, which gives more flexibility.
%
% INPUT/OUTPUT ===============================================================
% OPTION .... Optional. If not provided the script will start with #1.
%             (1) c=u       and u_0 = sin(x).
%             (2) c=u       and u_0 = exp(-(x-pi)^2).
%             (3) c=(1-2*u) and u_0 = 0.5*(1-sin(x)).
%             (4) c=(1-2*u) and u_0 = 0.5*(1-sin(x))+0.6*exp(-3*(x-2)^2).
%             (5) c=(1-2*u) and u_0 = Zig-Zag.
%             (6) c=u       and u_0 = "wavy".
% Fig3D ..... Handle for the figure with the 3D plot. \ OPTIONAL.
% FigSL ..... Handle for the figure with the slider.  /
%
% CHANGES RELATIVE TO PRIOR VERSIONS =========================================
% version 00 demoWBRch
% --- Show is controlled entirely from the keyboard.
% --- Cannot go back in time evolution. One shot deal.
% version 01 demoWBRch_v01
% --- Added slider to control which plot is shown.
% --- Improved looks of the figure windows (better labels, etc).
% --- Can skip the options menu by directly calling a choice via an input.
% version 02 demoWBRch_v02
% --- Added output with handles for figures.
% --- Added control button to switch between movie show and slider show.
% --- Added example button to switch between examples.
% --- Added close   button to stop demo.
% --- Added help window.
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Copyright 1999-2008 Massachusetts Institute of Technology
% Version 00 by Rodolfo R. Rosales  02-25-1999
% Version 01 by Rodolfo R. Rosales  03-24-1999
% Version 02 by Rodolfo R. Rosales  03-27-2002
% Update     by Rodolfo R. Rosales  03-10-2008.
% 
% Permission is hereby granted, without payment, to copy this software
% and its documentation, if any,  for non-profit academic and research
% purposes,  provided that the above copyright notice, this paragraph,
% and the following three paragraphs appear in all copies of this
% software. Use of this software constitutes acceptance of these terms
% and conditions.
%
% IN NO EVENT SHALL MIT, OR THE AUTHOR, BE LIABLE TO ANY PARTY FOR
% DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
% ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
% IF MIT, OR THE AUTHOR, HAVE BEEN ADVISED OF THE POSSIBILITY OF SUCH
% DAMAGE.
%
% MIT, AND THE AUTHOR, SPECIFICALLY DISCLAIM ANY EXPRESS OR IMPLIED
% WARRANTIES INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
% MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, AND
% NON-INFRINGEMENT.
%
% THIS SOFTWARE IS PROVIDED "AS IS." MIT, OR THE AUTHOR, HAVE NO
% OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS,
% OR MODIFICATIONS.
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
FLAG = 0;% --- Tells if figures already there, or not. Used by OPTION button.
% ============================================================================
% Notice that the user calls the script with either no arguments, or just one
% (the example number). In these cases the 1-st if-statement below is skipped.
%
% This first part ("comand and control" part of the script) works as follows:
% --- The buttons and slider call the script with TWO arguments (OPTION, N).
% --- N=0 corresponds to a call from the slider, with OPTION = slider handle.
%     Then the action is turned over to the sub-script demoWBRch_v02_SLD, to
%     update the plot. The UserData for the slider has all the info needed.
% --- N=1 corresponds to a call by the Movie button (OPTION = button handle).
%     Then:  Note that, for this button UserData = [Max value for slider,
%                                                   Slider handle,
%                                                   Play flag].
%     First:  The button is turned into the Stop button.
%             When the button is set to Stop,  Play = 1.
%             When the button is set to Movie, Play = 0.
%     Second: The slides are played as long as Play = 1. Each time the slider
%             value is upgraded by one, and the subscript demoWBRch_v02_SLD
%             is then used to upgrade the plot.
% --- N=2 corresponds to a call by the Stop button. Then the button is turned
%     back to the Movie button (this changes Play to 0, and stops movie).
% --- N=3 corresponds to a call by the example button (OPTION = button handle)
%     to change example. For this button UserData = [3D figure handle,
%                                                    Slider figure handle,
%                                                    Slider handle,
%                                                    Movie/Stop button handle,
%                                                    Current example #,
%                                                    Number of examples]
%     Then the following actions are taken:
%     First:  change the variable FLAG (set to zero at code start) to 1.
%     Second: upgrade example to next one.
%
%     The variable FLAG works as follows: at the end of this section, if the
%     script was not terminated by a return,   the variable OPTION will have
%     the example number to be done and:
%     If FLAG = 0, the code will create new figure windows.
%        FLAG = 1, the old figure windows will be used.
% ============================================================================
%
if nargin == 2
   if N == 0 % Show plot selected by slider.
      demoWBRch_v02_SLD(OPTION)
      return
   elseif N == 1 % Play as movie.
      BtnHndl = OPTION;
      BtnStrg = 'Stop';
      BtnColr = 'y';
      UsrData = get(BtnHndl, 'UserData');
      UsrData = [UsrData(1:2), 1];
      BtnCBck = ['demoWBRch_v02(', num2str(BtnHndl, '%37.30e') , ', 2)'];
      %
      set(BtnHndl, ...
          'UserData',         UsrData, ...
          'CallBack',         BtnCBck, ...
          'BackGroundColor',  BtnColr, ...
          'String',           BtnStrg);
      %
      SlrHndl = UsrData(2); % Slider handle
      Play = 1;
      while Play == 1
         %
         % This next if is so, when the close button closes all figures,
         % the movie stops and does not attempt to continue (generating
         % an error message.
         %
         if ~ishandle(BtnHndl)
            return
         end
         UsrData = get(BtnHndl, 'UserData');
         SlrMval = UsrData(1); % Max value slider
         Play    = UsrData(3);
         value   = get(SlrHndl, 'Value') + 1;
         if value > SlrMval
            value = 1;
            t0 = clock; while etime(clock, t0) < 0.75; end
         end
         set(SlrHndl, 'Value', value);
         demoWBRch_v02_SLD(SlrHndl)
         drawnow
         t0 = clock; while etime(clock, t0) < 0.07; end
      end
      return
   elseif N == 2 % Stop playing as movie.
      BtnHndl = OPTION;
      BtnStrg = 'Movie';
      BtnColr = 'c';
      UsrData = get(BtnHndl, 'UserData');
      UsrData = [UsrData(1:2), 0];
      BtnCBck = ['demoWBRch_v02(', num2str(BtnHndl, '%37.30e') , ', 1)'];
      %
      set(BtnHndl, ...
          'UserData',         UsrData, ...
          'CallBack',         BtnCBck, ...
          'BackGroundColor',  BtnColr, ...
          'String',           BtnStrg);
      return
   elseif N == 3 % Call to change the example being shown.
      BtnHnd2 = OPTION;
      UsrData = get(BtnHnd2, 'UserData');
      Fig3D   = UsrData(1);
      FigSL   = UsrData(2);
      SlrHndl = UsrData(3);
      BtnHndl = UsrData(4);
      OPTION  = UsrData(5);
      Noption = UsrData(6);
      FLAG    = 1;
      OPTION  = (OPTION < Noption)*OPTION + 1;
      %
      UsrData(5) = OPTION;
      set(BtnHnd2, 'UserData', UsrData, ...
                   'String',   ['Example #', num2str(OPTION)])
   end
end
%
% ============================================================================
% If FLAG = 0, a help window is setup.
% ============================================================================
%
if FLAG == 0
   HWinS1 = [ ...
         '                                                    '; ...
         ' SOLUTION BY CHARACTERISTICS of:                    '; ...
         '                                                    '; ...
         '          u_t + c(u)*u_x = 0, u(x,0) = u_0,         '; ...
         '                                                    '; ...
         ' with periodic boundary conditions in 0 < x < 2*pi. '; ...
         '                                                    '; ...
         ' These are the AVAILABLE EXAMPLES:                  '; ...
         '                                                    '; ...
         ' (1)  c=u        and  u_0 = sin(x).                 '; ...
         '                                                    '; ...
         ' (2)  c=u        and  u_0 = exp(-(x-pi)^2).         '; ...
         '                                                    '; ...
         ' (3)  c=(1-2*u)  and  u_0 = 0.5*(1-sin(x)).         '; ...
         '                                                    '; ...
         ' (4)  c=(1-2*u)  and  u_0 = 0.5*(1-sin(x)) +        '; ...
         '                            0.6*exp(-3*(x-2)^2).    '; ...
         '                                                    '; ...
         ' (5)  c=(1-2*u)  and  u_0 = Zig-Zag.                '; ...
         '                                                    '; ...
         ' (6)  c=u        and  u_0 = "wavy".                 '; ...
         '                                                    '];
   %
   helpwin(HWinS1)
   set(0, 'ShowHiddenHandles', 'on')
   HWinHdl = get(0, 'CurrentFigure');
   HWinCld = get(HWinHdl, 'Children');
   Nchild  = length(HWinCld);
   HWinPos = get(HWinHdl, 'Position');
   ScreenS = get(0, 'ScreenSize');
   HWinPos(1) = ScreenS(3)/50;
   set(HWinHdl, 'Position', HWinPos);
   set(HWinCld(Nchild), 'FontWeight', 'bold', 'FontSize', 11)
   for n=1:Nchild-1
      set(HWinCld(n), 'Visible', 'off')
   end
end
%
% ============================================================================
% If the script is called without any arguments, then example # 1 is selected.
% ============================================================================
%
if nargin == 0
   OPTION = 1;
end
%
% ============================================================================
% --- NEXT the nitial Conditions given by the various OPTIONS are evaluated.
%
% The variables are as follows:
%
% eta:   (1 x Ne) array with the characteristic "labels" in x=c*t+eta.
%        Equally spaced in [0 2*pi], with eta(1) = 0 and eta(Ne) = 2*pi.
% deta:  Equal spacing of the labels in eta.
% F:     (1 x Ne) array with the solution u --- u = F(eta).
% G:     (1 x Ne) array with the characteristic speeds, G = c(F(eta))
% ts:    (Nt x 1) array with the (equally spaced) time mesh.
% dt:    Equal spacing in the time mesh in ts.
% tmax:  Last time computed; t(1) = 0 and t(Nt) = tmax.
% umax:  Maximum value of the solution.
% umin:  Minimum value of the solution.
% xs:    (Nt x Ne) array with characteristics: xs(i,j)=G(j)*ts(i)+eta(j)
% us:    (Nt x Ne) array with solution: u(i,j) = F(j).
% Ne:    Number of points in space grid. .............. PARAMETER: can change.
% Nt:    Number of points in time grid. ............... PARAMETER: can change.
%
% Nice_view: String with a nice view setting for 3D plot.
% TITL:      String with title for plots.
%
% NOTE THAT SCHEME IS SET UP SO ADDING MORE OPTIONS IS EASY.
%
% ============================================================================
%
if OPTION == 1
   Ne = 101;   Nt = 51;   deta=2*pi/(Ne-1);   eta=0:deta:2*pi;
   TITL = 'c = u  and  u_0 = sin(x).';
   F = sin(eta);   G = F;   tmax = 3;   umax = 1;   umin = -1;
   Nice_view = 'view(200,20)';
elseif OPTION == 2
   Ne = 101;   Nt = 51;   deta=2*pi/(Ne-1);   eta=0:deta:2*pi;
   TITL = 'c = u  and  u_0 = exp(-(x-pi)^2).';
   F = exp(-(eta-pi).^2);   G = F;   tmax = 3;   umax = 1;   umin = 0;
   Nice_view = 'view(200,20)';
elseif OPTION == 3
   Ne = 101;   Nt = 51;   deta=2*pi/(Ne-1);   eta=0:deta:2*pi;
   TITL = 'c = 1 - 2u  and  u_0 = 0.5 (1-sin(x)).';
   F = 0.5*(1-sin(eta));   G = 1-2*F;   tmax = 3;   umax = 1;   umin = 0;
   Nice_view = 'view(200,20)';
elseif OPTION == 4
   Ne = 101;   Nt = 53;   deta=2*pi/(Ne-1);   eta=0:deta:2*pi;
   TITL = 'c = 1 - 2u  and  u_0 = 0.5 (1-sin(x)) + 0.6exp (-3(x-2)^2).';
   F = 0.5*(1 - sin(eta)) + 0.6*exp( - 3*(eta - 2).^2);   G = 1 - 2*F;
   tmax = 4.5;   umax = 1;   umin = 0;
   Nice_view = 'view(200,20)';
elseif OPTION == 5
   Ne = 101;   Nt = 46;   deta=2*pi/(Ne-1);   eta=0:deta:2*pi;
   TITL = 'c = 1 - 2u  and  u_0 = Zig-Zag.';
   F = min(abs(eta - pi/2), abs(eta - 5*pi/2))/pi;   G = 1 - 2*F;
   tmax = 1.5*pi;   umax = 1;   umin = 0;
   Nice_view = 'view(200,20)';
elseif OPTION == 6
   Ne = 101;   Nt = 51;   deta=2*pi/(Ne-1);   eta=0:deta:2*pi;
   TITL = 'c = u  and  u_0 = "wavy".';
   F = sin(eta) + 0.2*sin(2*eta) + 0.4*sin(3*eta);   G = F;
   tmax = 4.6;   umax = 1.2;   umin = - 1.2;
   Nice_view = 'view(200,20)';
end
%
Noption = 6; % --- Number of options.
%
dt  = tmax/(Nt-1);
ts  = (0:dt:tmax)';
UNO = ones(Nt, 1);
xs  = ts*G + UNO*eta;
us  = UNO*F;
%
% ============================================================================
% Below the two figures for the solution are created (if FLAG = 0), and then
% the solution is plotted (in the 3D case), or is setup for the slide show.
%
% OF THE PARAMETERS USED ABOVE IN CREATING THE SOLUTIONS, ONLY THE FOLLOWING
% ARE USED BY THE REST OF THE CODE:
%
% F:     (1 x Ne) array with the solution u --- u = F(eta).
%        The characteristic labels eta are no longer used, though.
% ts:    (Nt x 1) array with the (equally spaced) time mesh.
% tmax:  Last time computed; t(1) = 0 and t(Nt) = tmax.
% umax:  Maximum value of the solution.
% umin:  Minimum value of the solution.
% xs:    (Nt x Ne) array with characteristics: xs(i,j)=G(j)*ts(i)+eta(j)
% us:    (Nt x Ne) array with solution: u(i,j) = F(j).
% Nice_view: String with a nive view setting for 3D plot.
% TITL:      String with title for plots.
%
% ============================================================================
%
% --- 3-D plot of solution.
%
if FLAG == 0
   Inpt.Name = '3D plot of solution by characteristics.';
   Inpt.Slid = 0;
   Inpt.dX = 0.05;
   Inpt.dY = 0.05;
   Inpt.fW = 0.70;
   Inpt.fH = 0.70;
   Inpt.aW = 0.75;
   Inpt.aH = 0.75;
   %
   Outpt = MakeFig311_v00(Inpt);
   Fig3D = Outpt.f;% --- 3D figure handle.
   hold on
   grid on
   whitebg(Outpt.f, 'k')
elseif FLAG == 1
   set(Fig3D, 'Visible', 'off')
   figure(Fig3D)
   hold off; cla; hold on
end
axis([0 2*pi 0 tmax umin umax])
eval(Nice_view);
title(['{\bf', TITL, '}'], 'FontSize', 18, 'Color', 'w')
xlabel('{\bf x}', 'FontSize', 20)
ylabel('{\bf t}', 'FontSize', 20)
zlabel('{\bf u }', 'FontSize', 20, 'Rotation', 0)
surf(xs, ts, us)
colormap(hsv)
shading interp
rotate3d
%
% --- Setup Slider Figure.
%
if FLAG == 0
   Inpt.Name = 'Evolution of Solution in time.';
   Inpt.Slid = 1;
   Inpt.SldH = 0.05;
   Inpt.valu = 1;
   Inpt.mini = 1;
   Inpt.maxi = length(ts);
   Inpt.step = [1/(length(ts)-1), 0.2];
   Inpt.cbck = '--';
   Inpt.dX = 0.10;
   Inpt.dY = 0.10;
   Inpt.fW = 0.70;
   Inpt.fH = 0.70;
   Inpt.aW = 0.70;
   Inpt.aH = 0.70;
   %
   Outpt = MakeFig311_v00(Inpt);
   FigSL = Outpt.f;% -------- Figure with slider handle.
   whitebg(Outpt.f, 'k')
   SlrHndl = Outpt.s;% ------ Slider handle.
   SlrMval = length(ts);% --- Max value of slider.
   set(SlrHndl, ...
      'CallBack', ['demoWBRch_v02(', num2str(SlrHndl, '%37.30e') , ', 0)'])
   UsrData.xs   = xs;
   UsrData.ts   = ts;
   UsrData.F    = F;
   UsrData.umin = umin;
   UsrData.umax = umax;
   UsrData.a    = Outpt.a;
   UsrData.TITL = TITL;
   set(SlrHndl, 'UserData', UsrData)
   %
   text('Units',               'Normalized', ...
        'Position',             [0.50, 1.00], ...
        'FontUnits',           'Normalized', ...
        'FontWeight',          'Bold', ...
        'FontSize',             0.09, ...
        'HorizontalAlignment', 'Center', ...
        'String',              'Use the slider to see the plots.')
   %
   text('Units',               'Normalized', ...
        'Position',             [0.50, 0.80], ...
        'FontUnits',           'Normalized', ...
        'FontWeight',          'Bold', ...
        'FontSize',             0.09, ...
        'HorizontalAlignment', 'Center', ...
        'String',              'Use the Movie/Stop button to')
   text('Units',               'Normalized', ...
        'Position',             [0.50, 0.70], ...
        'FontUnits',           'Normalized', ...
        'FontWeight',          'Bold', ...
        'FontSize',             0.09, ...
        'HorizontalAlignment', 'Center', ...
        'String',              'start/stop')
   text('Units',               'Normalized', ...
        'Position',             [0.50, 0.60], ...
        'FontUnits',           'Normalized', ...
        'FontWeight',          'Bold', ...
        'FontSize',             0.09, ...
        'HorizontalAlignment', 'Center', ...
        'String',              'the time evolution movie.')
   %
   text('Units',               'Normalized', ...
        'Position',             [0.50, 0.40], ...
        'FontUnits',           'Normalized', ...
        'FontWeight',          'Bold', ...
        'FontSize',             0.09, ...
        'HorizontalAlignment', 'Center', ...
        'String',              'Use the Example button to')
   text('Units',               'Normalized', ...
        'Position',             [0.50, 0.30], ...
        'FontUnits',           'Normalized', ...
        'FontWeight',          'Bold', ...
        'FontSize',             0.09, ...
        'HorizontalAlignment', 'Center', ...
        'String',              'change to next example.')
   %
   text('Units',               'Normalized', ...
        'Position',             [0.50, 0.10], ...
        'FontUnits',           'Normalized', ...
        'FontWeight',          'Bold', ...
        'FontSize',             0.09, ...
        'HorizontalAlignment', 'Center', ...
        'String',              'Use the Close button to')
   text('Units',               'Normalized', ...
        'Position',             [0.50, 0.00], ...
        'FontUnits',           'Normalized', ...
        'FontWeight',          'Bold', ...
        'FontSize',             0.09, ...
        'HorizontalAlignment', 'Center', ...
        'String',              'leave the demo.')
   %
   axis off
elseif FLAG == 1
   SlrMval      = length(ts);% --- Max value of slider.
   UsrData      = get(SlrHndl, 'UserData');
   UsrData.xs   = xs;
   UsrData.ts   = ts;
   UsrData.F    = F;
   UsrData.umin = umin;
   UsrData.umax = umax;
   UsrData.TITL = TITL;
   set(SlrHndl,                            ...
       'Value',      1,                    ...
       'Max',        SlrMval,              ...
       'SliderStep', [1/(SlrMval-1), 0.2], ...
       'UserData',   UsrData)
   demoWBRch_v02_SLD(SlrHndl)
   set(Fig3D, 'Visible', 'on')
end
%
% --- Setup Button for Movie Show.
%
if FLAG == 0
   BtnStrg = 'Movie';
   BtnColr = 'c';
   BtnHndl = uicontrol( ...
             'Style',           'Pushbutton', ...
             'FontUnits',       'Normalized', ...
             'FontSize',         0.5, ...
             'FontWeight',      'bold', ...
             'BackGroundColor',  BtnColr, ...
             'Interruptible',   'on', ...
             'Units',           'Normalized', ...
             'Position',         [0.78 0.06 0.2 0.07], ...
             'String',           BtnStrg);
   UsrData = [SlrMval, SlrHndl, 0];
           % [Max value slider, Slider handle, Play]
   BtnCBck = ['demoWBRch_v02(', num2str(BtnHndl, '%37.30e') , ', 1)'];
   set(BtnHndl, 'UserData', UsrData, 'CallBack', BtnCBck)
elseif FLAG == 1
   UsrData    = get(BtnHndl, 'UserData');
   UsrData(1) = SlrMval;
   set(BtnHndl, 'UserData', UsrData)
end
%
% --- Setup Button for Example choice.
%
if FLAG == 0
   BtnStrg = ['Example #', num2str(OPTION)];
   BtnColr = 'r';
   BtnHnd2 = uicontrol( ...
             'Style',           'Pushbutton', ...
             'FontUnits',       'Normalized', ...
             'FontSize',         0.5, ...
             'FontWeight',      'bold', ...
             'BackGroundColor',  BtnColr, ...
             'Interruptible',   'on', ...
             'Units',           'Normalized', ...
             'Position',         [0.02 0.06 0.32 0.07], ...
             'String',           BtnStrg);
   UsrData = [Fig3D FigSL SlrHndl BtnHndl OPTION Noption];
   BtnCBck = ['demoWBRch_v02(', num2str(BtnHnd2, '%37.30e') , ', 3)'];
   set(BtnHnd2, 'UserData', UsrData, 'CallBack', BtnCBck)
elseif FLAG == 1
    UsrData = [Fig3D FigSL SlrHndl BtnHndl OPTION Noption];
    set(BtnHnd2, 'UserData', UsrData)
end
%
% --- Setup Close Buttons.
%
if FLAG == 0
   BtnStrg = 'Close';
   BtnCBck = ['H = [', num2str(Fig3D), ' , '              ...
                     , num2str(FigSL), ' , '              ...
                     , num2str(HWinHdl, '%37.30e'), ']; ' ...
              'if ishandle(H(1)); close(H(1)); end; '     ...
              'if ishandle(H(2)); close(H(2)); end; '     ...
              'if ishandle(H(3)); close(H(3)); end; '];
   %
   % In Slider Figure.
   %
   uicontrol( ...
             'Style',           'Pushbutton', ...
             'FontUnits',       'Normalized', ...
             'FontSize',         0.5, ...
             'FontWeight',      'bold', ...
             'BackGroundColor', 'w', ...
             'Units',           'Normalized', ...
             'Position',         [0.04 0.95 0.15 0.04], ...
             'CallBack',         BtnCBck, ...
             'String',           BtnStrg)
   %
   % In 3D Figure.
   %
   set(Fig3D, 'Visible', 'off')
   figure(Fig3D)
   uicontrol( ...
             'Style',           'Pushbutton', ...
             'FontUnits',       'Normalized', ...
             'FontSize',         0.5, ...
             'FontWeight',      'bold', ...
             'BackGroundColor', 'w', ...
             'Units',           'Normalized', ...
             'Position',         [0.04 0.95 0.15 0.04], ...
             'CallBack',         BtnCBck, ...
             'String',           BtnStrg)
   figure(FigSL)
   set(Fig3D, 'Visible', 'on')
   %
   % In Help Window.
   %
   set(HWinHdl, 'Visible', 'off')
   figure(HWinHdl)
   uicontrol( ...
             'Style',           'Pushbutton', ...
             'FontUnits',       'Normalized', ...
             'FontSize',         0.5, ...
             'FontWeight',      'bold', ...
             'BackGroundColor', 'y', ...
             'Units',           'Normalized', ...
             'Position',         [0.04 0.90 0.16 0.08], ...
             'CallBack',         BtnCBck, ...
             'String',           BtnStrg)
   figure(FigSL)
   set(HWinHdl, 'Visible', 'on')
end
%
if nargout == 0; clear Fig3D FigSL; end
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function demoWBRch_v02_SLD(SliderHandle)
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
UsrD = get(SliderHandle, 'UserData');
xs   = UsrD.xs;
ts   = UsrD.ts;
F    = UsrD.F;
umin = UsrD.umin;
umax = UsrD.umax;
a    = UsrD.a;
TITL = UsrD.TITL;
n    = round(get(SliderHandle, 'Value'));
%
axes(a);
hold off
plot(xs(n, :), F, '-c', 'LineWidth', 3)
hold on;
grid on;
zoom on;
set(a, 'FontWeight', 'Bold', ...
       'FontSize',    16, ...
       'LineWidth',   2, ...
       'Box',         'on')
TIME = num2str(ts(n));
axis([0 2*pi umin umax])
title(['{\bf', TITL, '}'], 'FontSize', 18)
xlabel('{\bf x }', 'FontSize', 20)
ylabel('{\bf u }', 'FontSize', 20, 'Rotation', 0, ...
       'HorizontalAlignment', 'right')
text('Units', 'Normalized', 'Position', [0.50, -0.15], ...
     'FontSize', 20, 'HorizontalAlignment', 'center', ...
     'String', ['{\bf Time t = ', TIME, '}'])
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%% END OF FILE
