function demoWBRch_v01(OPTION, N)
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% function demoWBRch_v01
% ____________________________________________________________________________
% 18.311 (Principles of Applied Mathematics). MIT, Winter 2008. R. R. Rosales.
%
% This demo shows various examples of the time evolution given by the equation
%
%                        u_t + c(u)*u_x = 0.
% 
% In all cases the evolution from initial conditions given for x in [0,  2*pi]
% are shown. The solution is calculated using CHARACTERISTICS
%
% See also evolveWBRch, which gives more flexibility.
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Copyright 1999-2008 Massachusetts Institute of Technology
% First version by Rodolfo R. Rosales  02-25-1999
% This  version by Rodolfo R. Rosales  03-24-1999
% Update        by Rodolfo R. Rosales  03-10-2008.
% 
% Permission is hereby granted, without payment, to copy this software
% and its documentation, if any,  for non-profit academic and research
% purposes,  provided that the above copyright notice, this paragraph,
% and the following three paragraphs appear in all copies of this
% software. Use of this software constitutes acceptance of these terms
% and conditions.
%
% IN NO EVENT SHALL MIT, OR THE AUTHOR, BE LIABLE TO ANY PARTY FOR
% DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
% ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
% IF MIT, OR THE AUTHOR, HAVE BEEN ADVISED OF THE POSSIBILITY OF SUCH
% DAMAGE.
%
% MIT, AND THE AUTHOR, SPECIFICALLY DISCLAIM ANY EXPRESS OR IMPLIED
% WARRANTIES INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
% MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, AND
% NON-INFRINGEMENT.
%
% THIS SOFTWARE IS PROVIDED "AS IS." MIT, OR THE AUTHOR, HAVE NO
% OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS,
% OR MODIFICATIONS.
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if nargin == 2
   demoWBRch_v01_SLD(OPTION)
   return
end
%
% The script works as follows: ***********************************************
% eta:   (1 x Ne) array with the characteristic "labels" in x=c*t+eta.
%        Equally spaced in [0 2*pi], with eta(1) = 0 and eta(Ne) = 2*pi.
% deta:  Equal spacing of the labels in eta.
% F:     (1 x Ne) array with the solution u --- u = F(eta).
% G:     (1 x Ne) array with the characteristic speeds, G = c(F(eta))
% ts:    (Nt x 1) array with the (equally spaced) time mesh.
% dt:    Equal spacing in the time mesh in ts.
% tmax:  Last time computed; t(1) = 0 and t(Nt) = tmax.
% umax:  Maximum value of the solution.
% umin:  Minimum value of the solution.
% xs:    (Nt x Ne) array with characteristics: xs(i,j)=G(j)*ts(i)+eta(j)
% us:    (Nt x Ne) array with solution: u(i,j) = F(j).
% Ne:    Number of points in space grid. .............. PARAMETER: can change.
% Nt:    Number of points in time grid. ............... PARAMETER: can change.
% Nskip: Skip used for frame by frame slide show. ..... PARAMETER: can change.
%
% NOTE THAT SCHEME IS SET UP SO ADDING MORE OPTIONS IS EASY.
%
if nargin == 0
   fprintf('\n SOLUTION BY CHARACTERISTICS of: \n')
   fprintf('\n               u_t + c(u)*u_x = 0, u(x,0) = u_0. \n')
   fprintf('\n You have the following OPTIONS: ')
   fprintf('\n (1) c=u       and u_0 = sin(x).')
   fprintf('\n (2) c=u       and u_0 = exp(-(x-pi)^2).')
   fprintf('\n (3) c=(1-2*u) and u_0 = 0.5*(1-sin(x)).')
   fprintf('\n (4) c=(1-2*u) and u_0 = 0.5*(1-sin(x))+0.6*exp(-3*(x-2)^2).')
   fprintf('\n (5) c=(1-2*u) and u_0 = Zig-Zag.')
   fprintf('\n (6) c=u       and u_0 = "wavy". \n')
   %
   OPTION = input(' Enter OPTION = ');
   if isempty(OPTION); OPTION = 1; end
   fprintf('\n')
end
%
% --- EVALUATE Initial Conditions given by the various OPTIONS ***************
%
if OPTION == 1
   Ne = 101;   Nt = 51;   Nskip = 5;   deta=2*pi/(Ne-1);   eta=0:deta:2*pi;
   TITL = 'c = u  and  u_0 = sin(x).';
   F = sin(eta);   G = F;   tmax = 3;   umax = 1;   umin = -1;
   Nice_view = 'view(200,20)';
elseif OPTION == 2
   Ne = 101;   Nt = 51;   Nskip = 5;   deta=2*pi/(Ne-1);   eta=0:deta:2*pi;
   TITL = 'c = u  and  u_0 = exp(-(x-pi)^2).';
   F = exp(-(eta-pi).^2);   G = F;   tmax = 3;   umax = 1;   umin = 0;
   Nice_view = 'view(200,20)';
elseif OPTION == 3
   Ne = 101;   Nt = 51;   Nskip = 5;   deta=2*pi/(Ne-1);   eta=0:deta:2*pi;
   TITL = 'c = 1 - 2u  and  u_0 = 0.5 (1-sin(x)).';
   F = 0.5*(1-sin(eta));   G = 1-2*F;   tmax = 3;   umax = 1;   umin = 0;
   Nice_view = 'view(200,20)';
elseif OPTION == 4
   Ne = 101;   Nt = 53;   Nskip = 4;   deta=2*pi/(Ne-1);   eta=0:deta:2*pi;
   TITL = 'c = 1 - 2u  and  u_0 = 0.5 (1-sin(x)) + 0.6exp (-3(x-2)^2).';
   F = 0.5*(1 - sin(eta)) + 0.6*exp( - 3*(eta - 2).^2);   G = 1 - 2*F;
   tmax = 4.5;   umax = 1;   umin = 0;
   Nice_view = 'view(200,20)';
elseif OPTION == 5
   Ne = 101;   Nt = 46;   Nskip = 5;   deta=2*pi/(Ne-1);   eta=0:deta:2*pi;
   TITL = 'c = 1 - 2u  and  u_0 = Zig-Zag.';
   F = min(abs(eta - pi/2), abs(eta - 5*pi/2))/pi;   G = 1 - 2*F;
   tmax = 1.5*pi;   umax = 1;   umin = 0;
   Nice_view = 'view(200,20)';
elseif OPTION == 6
   Ne = 101;   Nt = 51;   Nskip = 5;   deta=2*pi/(Ne-1);   eta=0:deta:2*pi;
   TITL = 'c = u  and  u_0 = "wavy".';
   F = sin(eta) + 0.2*sin(2*eta) + 0.4*sin(3*eta);   G = F;
   tmax = 4.6;   umax = 1.2;   umin = - 1.2;
   Nice_view = 'view(200,20)';
end
%
%
%
dt  = tmax/(Nt-1);
ts  = (0:dt:tmax)';
UNO = ones(Nt, 1);
xs  = ts*G + UNO*eta;
us  = UNO*F;
%
% --- 3-D plot of solution.
%
Inpt.Name = '3D plot of solution by characteristics.';
Inpt.Slid = 0;
Inpt.dX = 0.05;
Inpt.dY = 0.05;
Inpt.fW = 0.70;
Inpt.fH = 0.70;
Inpt.aW = 0.75;
Inpt.aH = 0.75;
%
Outpt = MakeFig311_v00(Inpt);
hold on
grid on
whitebg(Outpt.f, 'k')
axis([0 2*pi 0 tmax umin umax])
eval(Nice_view);
title(['{\bf', TITL, '}'], 'FontSize', 18)
xlabel('{\bf x}', 'FontSize', 20)
ylabel('{\bf t}', 'FontSize', 20)
zlabel('{\bf u }', 'FontSize', 20, 'Rotation', 0)
surf(xs, ts, us)
colormap(hsv)
shading interp
rotate3d
%
% --- Setup Slider Figure.
%
Inpt.Name = 'Evolution of Solution in time.';
Inpt.Slid = 1;
Inpt.SldH = 0.05;
Inpt.valu = 1;
Inpt.mini = 1;
Inpt.maxi = length(ts);
Inpt.step = [1/(length(ts)-1), 0.2];
Inpt.cbck = '--';
Inpt.dX = 0.10;
Inpt.dY = 0.10;
Inpt.fW = 0.70;
Inpt.fH = 0.70;
Inpt.aW = 0.70;
Inpt.aH = 0.70;
%
Outpt = MakeFig311_v00(Inpt);
whitebg(Outpt.f, 'k')
set(Outpt.s, ...
   'CallBack', ['demoWBRch_v01(', num2str(Outpt.s, '%37.30e') , ', 0)'])
UsrData.xs   = xs;
UsrData.ts   = ts;
UsrData.F    = F;
UsrData.umin = umin;
UsrData.umax = umax;
UsrData.a    = Outpt.a;
UsrData.TITL = TITL;
set(Outpt.s, 'UserData', UsrData)
text('Units',               'Normalized', ...
     'Position',             [0.50, 0.50], ...
     'FontUnits',           'Normalized', ...
     'FontWeight',          'Bold', ...
     'FontSize',             0.09, ...
     'HorizontalAlignment', 'Center', ...
     'String',              'Use the slider to see the plots.')
axis off

function demoWBRch_v01_SLD(SliderHandle)

UsrD = get(SliderHandle, 'UserData');
xs   = UsrD.xs;
ts   = UsrD.ts;
F    = UsrD.F;
umin = UsrD.umin;
umax = UsrD.umax;
a    = UsrD.a;
TITL = UsrD.TITL;
n    = round(get(SliderHandle, 'Value'));
%
axes(a);
hold off
plot(xs(n, :), F, '-c', 'LineWidth', 3)
hold on;
grid on;
zoom on;
set(a, 'FontWeight', 'Bold', ...
       'FontSize',    16, ...
       'LineWidth',   2, ...
       'Box',         'on')
TIME = num2str(ts(n));
axis([0 2*pi umin umax])
title(['{\bf', TITL, '}'], 'FontSize', 18)
xlabel('{\bf x }', 'FontSize', 20)
ylabel('{\bf u }', 'FontSize', 20, 'Rotation', 0, ...
       'HorizontalAlignment', 'right')
text('Units', 'Normalized', 'Position', [0.50, -0.15], ...
     'FontSize', 20, 'HorizontalAlignment', 'center', ...
     'String', ['{\bf Time t = ', TIME, '}'])

%
%% END OF FILE
