function EIKO3Ddemo(Ex, SliderHandle)
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EIKO3Ddemo
% ____________________________________________________________________________
% 18.311 (Principles of Applied Mathematics). MIT, Winter 2008. R. R. Rosales.
%
% Demo with some examples illustrating the evolution of wave fronts,  as given
% by the Eikonal equation  (Phi_x)^2 + (Phi_y)^2 + (Phi_z)^2 = 1 --- motion at
% unit speed, normal to the wavefront.
%
% The rays and caustics are also shown.
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Copyright 2001-2008 Massachusetts Institute of Technology
% Version 01 by Rodolfo R. Rosales  03-27-2001.
% Update     by Rodolfo R. Rosales  03-15-2008.
% 
% Permission is hereby granted, without payment, to copy this software
% and its documentation, if any,  for non-profit academic and research
% purposes,  provided that the above copyright notice, this paragraph,
% and the following three paragraphs appear in all copies of this
% software. Use of this software constitutes acceptance of these terms
% and conditions.
%
% IN NO EVENT SHALL MIT, OR THE AUTHOR, BE LIABLE TO ANY PARTY FOR
% DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
% ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
% IF MIT, OR THE AUTHOR, HAVE BEEN ADVISED OF THE POSSIBILITY OF SUCH
% DAMAGE.
%
% MIT, AND THE AUTHOR, SPECIFICALLY DISCLAIM ANY EXPRESS OR IMPLIED
% WARRANTIES INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
% MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, AND
% NON-INFRINGEMENT.
%
% THIS SOFTWARE IS PROVIDED "AS IS." MIT, OR THE AUTHOR, HAVE NO
% OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS,
% OR MODIFICATIONS.
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if nargin == 2% --- Two inputs are used by the slider callback.
   EIKO3Ddemo_plot(SliderHandle);   return;   end
% ____________________________________________________________________________
% --- If the user does not give a selection, this goes through the list %%%%%%
%     of available example files, allowing the user to pick an example. %%%%%%
%
if nargin == 0;   fprintf('\n These are the currently available examples:')
   ExistNN = 2;   nn = 0;
   while ExistNN == 2
      nn = nn+1;   Example = ['EIKO3Dexample', num2str(nn,'%02.0f')];
      ExistNN = exist(Example);
      if ExistNN == 2;   eval([Example, '(0)']);   end
   end
   fprintf('\n Select 0 to stop now.\n')
   Ex = input(' Select an example and enter your selection number: Ex = ');
   if isempty(Ex);   EIKO3Ddemo;   return;   end
end
% ____________________________________________________________________________
% --- Check that selected example exists. %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
if Ex == 0; return; end
if exist(['EIKO3Dexample', num2str(Ex,'%02.0f')]) ~= 2;   EIKO3Ddemo;   return
end
% ____________________________________________________________________________
% --- If an example choice is given, execute the example file. %%%%%%%%%%%%%%%
%
eval(['Output = EIKO3Dexample', num2str(Ex,'%02.0f'), ';'])
exa = Output.exa;
txt = Output.txt;
Nsl = Output.Nsl;
Nbe = Output.Nbe;
% ____________________________________________________________________________
% --- Setup slider figure. %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
Inpt.Name = 'Eikonal wavefront evolution / Caustics.';     Inpt.dX = 0.05;
Inpt.Slid = 1;                                             Inpt.dY = 0.06;
Inpt.SldH = 0.05;                                          Inpt.fW = 0.88;
Inpt.valu = 1;   Inpt.mini = 1;   Inpt.maxi = Nsl+Nbe+1;   Inpt.aW = 0.70;
Inpt.step = [1/(Nsl+Nbe), max(0.2, 2/(Nsl+Nbe))];          Inpt.fH = 0.88;
Inpt.cbck = '--';                                          Inpt.aH = 0.70;
%
Outpt = MakeFig311_v00(Inpt);   whitebg(Outpt.f, 'k');   Output.a = gca;
set(Outpt.s, ...
   'CallBack', ['EIKO3Ddemo(0, ', num2str(Outpt.s, '%37.30e') , ')'], ...
   'UserData', Output)
%
EIKO3DwelcomeScreenPlot(txt, exa)
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function EIKO3DwelcomeScreenPlot(txt, exa)  % Welcome text in plot windows.
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
Bo = 'Bold';                FS = 'FontSize';     No = 'Normalized';
Ce = 'Center';              FU = 'FontUnits';    Po = 'Position';
Co = 'Color';               FW = 'FontWeight';   Un = 'Units';
HA = 'HorizontalAlignment';
text(Un, No, Po, [0.50, 0.96], FU, No, FW, Bo, FS, 0.08, HA, Ce, ...
     'String', 'Eikonal equation')
text(Un, No, Po, [0.50, 0.83], FU, No, FW, Bo, FS, 0.08, HA, Ce, ...
     'String', ' (\Phi_x)^2 + (\Phi_y)^2 + (\Phi_z)^2 = 1')
text(Un, No, Po, [0.50, 0.70], FU, No, FW, Bo, FS, 0.08, HA, Ce, ...
     'String', 'wavefront evolution.')
text(Un, No, Po, [0.50, 0.56], FU, No, FW, Bo, FS, 0.07, HA, Ce, Co, 'c', ...
     'String', exa)
text(Un, No, Po, [0.50, 0.46], FU, No, FW, Bo, FS, 0.045, HA, Ce, Co, 'c', ...
     'String', txt)
text(Un, No, Po, [0.50, 0.33], FU, No, FW, Bo, FS, 0.05, HA, Ce, ...
     'String', 'Caustic: shown at the end of the slider range.')
text(Un, No, Po, [0.50, 0.26], FU, No, FW, Bo, FS, 0.05, HA, Ce, ...
     'String', 'Use the mouse to rotate the image.')
text(Un, No, Po, [0.50, 0.00], FU, No, FW, Bo, FS, 0.07, HA, Ce, Co, 'y', ...
     'String', 'Use the slider to see the plots.')
axis off
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function EIKO3Ddemo_plot(SliderHandle) % Called by slider to plot.
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
Bo = 'Bold';        FW = 'FontWeight';            Po = 'Position';
Co = 'Color';       HA = 'HorizontalAlignment';   Ri = 'Right';
FS = 'FontSize';    LW = 'LineWidth';             St = 'String';
FU = 'FontUnits';   No = 'Normalized';            Un = 'Units';
%
Output = get(SliderHandle, 'UserData');
%
xmin = Output.xmin;   exa = Output.exa;   X  = Output.X;
xmax = Output.xmax;   txt = Output.txt;   Y  = Output.Y;
ymin = Output.ymin;   Nsl = Output.Nsl;   Z  = Output.Z;
ymax = Output.ymax;   Nbe = Output.Nbe;   Xe = Output.Xe;
zmin = Output.zmin;   ver = Output.ver;   Ye = Output.Ye;
zmax = Output.zmax;   T   = Output.T;     Ze = Output.Ze;
%
n = round(get(SliderHandle, 'Value'));
%
a = Output.a;   axes(a);   cla reset
if n == 1;
   EIKO3DwelcomeScreenPlot(txt, exa)
elseif (1 < n) & (n <= Nsl+1)
   TIME = num2str(T(n-1), '%06.4f');
   hold on;   grid on;   rotate3d on
   C = Z{1}; J = find(C == inf); C(J) = zmax;
   surf(X{n-1}, Y{n-1}, Z{n-1}, C)
   colormap jet
   shading flat
   set(a, FW, Bo, FU, No, FS, 0.035, LW, 2, 'Box', 'on')
   title(['{\bf Wave front for t = ', TIME, '}'], FU, No, FS, 0.06, 'Co', 'y')
   xlabel('{\bf x}', FU, No, FS, 0.08)
   ylabel('{\bf y}', FU, No, FS, 0.08)
   zlabel('{\bf z}', FU, No, FS, 0.08, 'Rotation', 0)
   axis([xmin xmax ymin ymax zmin zmax])
   view(ver)
   text(Un, No, Po, [-0.20 -0.25], FU, No, FS, 0.050, Co, 'y', 'Rotation', ...
        90, St, ['{\bf ', txt, '}'])
   text(Un, No, Po, [ 1.10  0.97], FU, No, FS, 0.055, 'Rotation', -90, St, ...
        '{\bf Use the mouse to rotate the image.}')
elseif Nsl+1 < n
   hold on;   grid on;   rotate3d on
   BRANCH = num2str(n-Nsl-1);
   C = Ze{n-Nsl-1}; J = find(C == inf); C(J) = zmax;
   surf(Xe{n-1-Nsl}, Ye{n-1-Nsl}, Ze{n-1-Nsl}, C)
   colormap jet
   shading flat
   set(a, FW, Bo, FU, No, FS, 0.035, LW, 2, 'Box', 'on')
   title(['{\bf Branch ', BRANCH, ' of the caustic.}'], ...
         FU, No, FS, 0.06, 'Co', 'y')
   xlabel('{\bf x}', FU, No, FS, 0.08)
   ylabel('{\bf y}', FU, No, FS, 0.08)
   zlabel('{\bf z}', FU, No, FS, 0.08, 'Rotation', 0)
   axis([xmin xmax ymin ymax zmin zmax])
   view(ver)
   text(Un, No, Po, [-0.20 -0.25], FU, No, FS, 0.050, Co, 'y', 'Rotation', ...
        90, St, ['{\bf ', txt, '}'])
   text(Un, No, Po, [ 1.10  0.97], FU, No, FS, 0.055, 'Rotation', -90, St, ...
        '{\bf Use the mouse to rotate the image.}')
end
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%% EOF
