function [x_c, rho_c, c_c, t_c] = strangeCFSM(Nh, rho_N, rho_M, alpha, FLAG)
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% function [x_c, rho_c, c_c, t_c] = strangeCFSM(Nh, rho_N, rho_M, alpha, FLAG)
% ____________________________________________________________________________
% 18.311 (Principles of Applied Mathematics). MIT, Winter 2008. R. R. Rosales.
%
%           Implements the ODE model for traffic flow introduced in the notes:
% "Simplest Car Following Traffic Flow Model".   These are posted in the 18311
% home page. The (nondimensional) model is:
%
%                    dx_n/dt = u_n = U(rho_n)   for n=1:N,
%
% where 0 < rho_n = epsilon/(x_(n+1) - x_n)) < 1  and the density rho_N at the
% leading car is given  (equivalently, the velocity u_N of this car is given).
% The car velocity function U is given by the simple (nondimensional) choice
%
%        U(rho) = min{ alpha^(-1), (1-rho)*[rho*(1-alpha)]^(-1) },
%
% where 0 < alpha < 1 --- SEE THE NOTES, (EXAMPLE 4.2) for more details.
% ____________________________________________________________________________
% The initial conditions are taken so that there is a hump in the density. The
% hump goes all the way from the density rho_N to the density rho_M, and it is
% located in the region -pi < x < 0. Specifically, AT TIME t=0, the density as
% a function of the nondimensional distance along the road has the form:
%
%                    rho_n(0) = rho_N + (rho_M - rho_N)*R(x_n),
% 
% where R = R(x) is continuous, vanishes outside of [-pi, 0]  and makes a hump
% (positive and symmetric) of height 1 for -pi < x < 0.
%
%  NOTE THAT,  IF rho_M < rho_N,  THE INITIAL PROFILE HAS A DIP (not a hump).
%
% The leading car (car number N) is at x = 0.   The hump comprises Nh cars ---
% i.e. car number p = (N+1-Nh) is at position x = -pi.     Thus cars numbers 1
% through p have initial density rho_N and are equally spaced, with
%
%                   x_1(0) = - {pi + (N-Nh)*epsilon/rho_N}.
%
% N is picked so there are enough cars to see all the "action".  The parameter
% epsilon is determined by the process explained in remark 5.1 of the notes.
%
% THE SOLUTION IS SHOWN VIA A MOVIE-LIKE SLIDE SHOW.
% ____________________________________________________________________________ 
% ____________________________________________________________________________
% Description of the outputs and required inputs:
% ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
% INPUTS
%  1) Nh    = Number of cars in the density disturbance hump at at t = 0.
%             Suggestion: 40 < Nh < 200.   For Nh too large, script runs slow.
%             For Nh not in [11, 1000], script uses Nh = 11.
%             NOTE: the actual number of cars used (i.e.: N) is larger than Nh
%             --- since the initial conditions add a "tail" to the hump.
%  2) rho_N = Nondimensional car density at the first car:     0 < rho_N <= 1.
%             For rho_N not in this range, script uses rho_N = 0.3.
%  3) rho_M = Value of the density at the hump center:         0 < rho_M <= 1.
%             For rho_M not in this range, script uses rho_M = 0.7.
%  4) alpha = Critical value for density in U(rho):       0.05 < alpha < 0.95.
%             For alpha not in this range, script uses alpha = 0.5.
%             NOTE: in principle one could have any 0 < alpha < 1.   But, when
%             alpha gets too close to the ends, the function U develops either
%             large values or large derivatives --- bad news for the numerics.
%  5) FLAG .. With FLAG = 1 code uses "normal" safety bounds (these are bounds
%                           on how many time steps, how many cars (in the tail
%                           before the bump), etc., are allowed).
%             With FLAG = 0 code uses looser safety bounds. May run slower.
%
% NOTE: alpha between rho_N and rho_M is needed for any shocks to form.
%
% OUTPUTS
% x_c ....... Ntime by N array with the car positions. Each row a time.
% rho_c ..... """"""""""""""""""""""""""""" density.   """"""""""""""""
% c_c ....... """"""""""""""""""""""""" wave speeds.   """"""""""""""""
% t_c ....... Ntime by 1 array with the times.
%
% NOTE: N and Ntime can be obtained using size(x_c).
%
% Code can be run without any outputs with the command:
%
% strangeCFSM(Nh, rho_N, rho_M, alpha, FLAG);
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% ********************** Adjustable parameters. ______________________________
%
 Nslides =   50;    %%%%%% Number of slides.
 Nlim_dt =  2e5;    %%%%%% Limit on the number of time steps. Safety.
%                          No more than Nlim_dt/Nh time steps can be taken.
 NtLim   =    5;    %%%%%% Biggest allowed ratio Nt/Nh ==> N <= (NtLim+1)*Nh.
 rNlim   = 0.02;    %%%%%% How close rN has to be to alpha before we deal with
%                          them as "equal" in the shock calculation.
 if FLAG == 0
    Nlim_dt = 1e7; NtLim = 10; rNlim = 0.01;
 end
%
% ###################### Safety checks. ______________________________________
%
 alpha_store = alpha;
 rN = rho_N;
 rM = rho_M;
 Nh = floor(Nh);
 if (Nh < 11)|(Nh > 1000);               Nh = 11;
    fprintf('\n WARNING: Nh out of bounds, will use Nh = 11. \n')
 end
 if (rN < 100*eps)|(rN >= 1);            rN = 0.3;
    fprintf('\n WARNING: rho_N out of bounds, will use rho_N = 0.3. \n')
 end
 if (rM < 100*eps)|(rM >= 1);            rM = 0.7;
    fprintf('\n WARNING: rho_M out of bounds, will use rho_M = 0.7. \n')
 end
 if (alpha <= 0.05)|(alpha >= 0.95);  alpha = 0.5;
    fprintf('\n WARNING: alpha out of bounds, will use alpha = 0.5. \n')
 end
 if abs(rN-alpha) <= rNlim+eps
    fprintf('\n WARNING: when alpha is too close to rho_N,   the time and')
    fprintf('\n          space scales over which the interesting features')
    fprintf('\n          occur are rather big.  Script may not be able to')
    fprintf('\n          encompass them.')
    if FLAG == 1
       fprintf('\n Try using FLAG = 0.')
    end
    fprintf('\n')
 end
%
% ###################### Do now hump (sine of car number) & epsilon. _________
%
% *** S1 and -S2 are the two values c can take.
%
 S1 = 1/alpha; S2 = 1/(1-alpha);
%
 xh = (0:1:(Nh-1))*pi/(Nh-1);   
 rh = rN + (rM-rN)*sin(xh);
 ch = (rh < alpha)*S1 - (rh > alpha)*S2;
 for n=2:Nh; xh(n) = xh(n-1) + 1/rh(n-1); end
 epsilon = pi/xh(Nh);
 xh = epsilon*xh - pi;
%
% ###################### Compute needed parameters. __________________________
%
 cN   = (rN < alpha)*S1 - (rN > alpha)*S2;
 uN   = min(1/alpha, (1-rN)/((1-alpha)*rN));
 cM   = (rM < alpha)*S1 - (rM > alpha)*S2;
%
% *** Set up time step for stability.  In this example (see notes, section 3):
%
%        for rho < alpha, nu=0             ===> tau_min = 0.
%        for rho > alpha,
%                   nu=1/((1-alpha)*rho^2) ===> tau_min = epsilon*(1-alpha)/2.
%
%     Euler scheme stability requires: dt <= 2*tau_min. For safety take:
%
 dt  = epsilon*(1-alpha)/2;
%
% *** Now we will ESTIMATE HOW FAR TO COMPUTE. Basically: the part of the bump
%     that is on the "other side" of alpha from r_N will be "sheared off", and
%     the bump profile will be flattened at alpha.  The extra area is added to
%     the other part, as follows:
%
% 01) Consider the case when rM > alpha > rN: --------------------------------
%
%     Define Xa and Xb by:  rho(Xa, 0) = rho(Xb, 0) = alpha,  Xa > Xb. The top
%     of the bump --- where rho(x, 0) > alpha --- is then on Xb < x < Xa.   We
%     talk about "top" (or bottom) to refer to this part of the rho profile.
%
%     The bottom two pieces move forward (rigidly) at speed  S1, while the top
%     moves (backward) at speed -S2. A shock then forms at the back.
%
%     When will the top vanish? Well, when the characteristic starting at Xa-0
%     (speed -S2) reaches the shock (time t = t_v).  From the point of view of
%     a coordinate frame moving at speed S1, the bottom does not move. In this
%     frame (let shock be at x_v at time t_v) we must have (conservation):
%           A1 = Integral_{x_v < x < Xb} (alpha - rho(x, 0))dx =
%           A2 = Integral_{Xb  < x < Xa} (rho(x, 0) - alpha)dx.
%     Note: convertion to fixed frame is ................ x_v ---> x_v + S1*t.
%
%     Estimate now:  (Xb-x_v)*beta >= A1 >= -(pi+x_v)*beta, (beta = alpha-rN).
%
%     In the moving frame the characteristic is  x = Xa -(S1+S2)*t.      Thus:
%     x_v = Xa - (S1+S2)*t_v  ===>
%
%        (S1+S2)*t_v = (Xa-x_v) <= -x_v <= (pi+A1/beta) = (pi+A2/beta)
%
% 02) Consider the case when rN > alpha > rM: --------------------------------
%
%     The situation is the same,  except that now it is the bottom part of the
%     "dip" that gets sheared and the shock appears ahead.   The same estimate
%     for t_v applies, with beta = rN - alpha.
%
% 03) Consider the case when (rN & rM > alpha) or (rN & rM > alpha) ----------
%
%     Nothing happens. Whole profile moves rigidly at speed S1 or -S2.
%
%     NOTE THAT WHEN beta is small (but not 0)  01) and 02) give big estimates
%     for x_v and t_v.   This because the bump is smeared in a very thin layer
%     to be flattened!  Thus, better treat rN ~ alpha as if they were equal!
%
% 04) We estimate A2 (using the approximation: dx = epsilon/rho)  and use this
%     to get a final time beyond t_v  --- with some safety is added to prevent
%     outrageously large t_f.
%
 beta = abs(rN - alpha);
 if beta < rNlim; beta = rNlim; end;
 A2  = sum((rN < alpha)*(rh > alpha).*(rh-alpha)./rh + ...
           (rN > alpha)*(rh < alpha).*(alpha-rh)./rh);
 A2  = epsilon*A2;
 t_f = (pi+A2/beta)/(S1+S2);
 if t_f > (Nlim_dt/Nh)*dt; t_f = (Nlim_dt/Nh)*dt;
    fprintf('\n WARNING: Limit on number of time steps implemented.')
    fprintf('\n          Calculation may not run long enough to see')
    fprintf('\n          all interesting details and/or the x range')
    fprintf('\n          may be improperly set.')
    fprintf('\n Try reducing the size of Nh or take a smaller alpha')
    fprintf('\n or reduce initial area between alpha and rho_M.')
    if FLAG == 1
       fprintf('\n OR: try using FLAG = 0.')
    end
    fprintf('\n')
 end
%
% *** Determine the time step between slides.
%
 Nstep = Nslides-1;
 dts   = t_f/Nstep;
%
% *** NOW WE ESTIMATE WERE TO PUT TRAILING CAR so we do not "run out of cars"
%     during the calculation.
% *** Bounds for where is bump at t=t_f: i.e.: xL < x < xR
%
 xR = cM*t_f + 0.2*pi;  % These two apply in the linear set up. (constant c). 
 xL = xR-pi - 0.4*pi;   % The NN*pi is because corners are smeared.    We use
%                       % cM to avoid problems when rN is too close to alpha.                              
 if (rM > alpha)&(alpha > (rN+rNlim))
    xR = S1*t_f;
    xL = xR-(pi+A2/beta);
 elseif (rM < alpha)&(alpha < (rN-rNlim))
    xL = -(pi+S2*t_f);
    xR = A2/beta - S2*t_f;
 end
%
% *** Thus we make the last car arrive at the bump at t_f, with some safety.
%
 xT  = min(xL-uN*t_f, -3.5);
 dxt = epsilon/rN; %%%%%%%%%%%%%%%%%% separation between cars in tail.
 xTlim = -pi - NtLim*Nh*dxt;
 if xT < xTlim; xT = xTlim;
    fprintf('\n WARNING: bound on how far behind  (the leading car) is the')
    fprintf('\n          trailing car implemented. There may not be enough')
    fprintf('\n          cars to see the full wave while it evolves:   the')
    fprintf('\n          "back end" may be lost.')
    fprintf('\n Try lowering the value of rho_N or decreasing the size of')
    fprintf('\n the initial area between alpha and rho_M. Moving alpha is')
    fprintf('\n best for this.')
    if FLAG == 1
       fprintf('\n OR: try using FLAG = 0.')
    end
    fprintf('\n')
 end
%
% ###################### Construct tail behind bump. _________________________
%
 Nt  = ceil((-pi-xT)/dxt);
 xt = (-pi-Nt*dxt):dxt:(-pi-dxt);
 rt = rN*ones(1, Nt);
 ct = cN*ones(1, Nt);
%
% ###################### Do now the initial data. ____________________________
%
 t = 0;
 x = [xt, xh];
 r = [rt, rh];
 c = [ct, ch];
 N = Nt + Nh;
%
% ###################### Start now calculation. ______________________________
%
 x_c   = x;
 rho_c = r;
 c_c   = c;
 t_c   = t;
%
 FN = 'FontName'; FS = 'FontSize'; HNB = 'HelveticaNarrowBold';
 LW = 'LineWidth'; MS = 'MarkerSize';
 EJES = [max(min(xL, -pi) - 1, xT), max(xR, 0)+1, ...
                       max(0, min(rN-0.1*(rM-rN), rM+0.1*(rM-rN))), ...
                       min(1, max(rN-0.1*(rM-rN), rM+0.1*(rM-rN)))];
%
% *** PLOT INITIAL DATA.
%
 figure
 whitebg('k')
 plot(x, r,  '-c', LW, 3)
 hold on
 plot([x(1), x(N)], [r(1), r(N)], '.r', MS, 20)
 plot([x(1), x(N)], [r(1), r(N)], 'og', MS, 10)
 plot([EJES(1), EJES(2)], [alpha, alpha], '--m')
 fprintf('\n Value of alpha plotted in magenta (if in range of plot).')
 axis(EJES)
 TEXT = num2str(t);
 title(['Nondimensional time t = ', TEXT], FN, HNB, FS, 20)
 ylabel('Nondimensional Car Density.', FN, HNB, FS, 18)
 xlabel('Nondimensional distance on road.', FN, HNB, FS, 18)
 hold off
 pause(0.5)
%
% *** DO INTEGRATION.
%
 for j=1:Nstep
    tt  = 0;
    while tt < dts
       ddt = ((dts-tt) > dt)*dt + ((dts-tt) <= dt)*(dts-tt);
       x   = x + ddt*(S1*(r<=alpha) + S2*(r>alpha).*(1-r)./r);
       r   = [epsilon./(x(2:N)-x(1:N-1)), rN];
       tt  = tt+ddt;
    end
    t = t+dts;
    c = (r < alpha)*S1 - (r > alpha)*S2;
    x_c   = [x_c; x];
    rho_c = [rho_c; r];
    c_c   = [c_c; c];
    t_c   = [t_c; t];
 end
%
% ###################### Slide show now. _____________________________________
%
 fprintf('\n _____________________________________________________________\n')
 fprintf('\n Ready to start slide show. TO START: press any key.\n')
 fprintf('\n Trailing & leading car positions: red dots with green circle.\n')
 fprintf('\n _____________________________________________________________\n')
 fprintf('\n')
 pause
 figure
 whitebg('k')
 for n=1:Nslides
    plot(x_c(n, :), rho_c(n, :),  '-c', LW, 3)
    hold on
    plot([x_c(n, 1), x_c(n, N)], [rho_c(n, 1), rho_c(n, N)], '.r', MS, 20)
    plot([x_c(n, 1), x_c(n, N)], [rho_c(n, 1), rho_c(n, N)], 'og', MS, 10)
    plot([EJES(1), EJES(2)], [alpha, alpha], '--m')
    axis(EJES)
    TEXT = num2str(t_c(n));
    title(['Nondimensional time t = ', TEXT], FN, HNB, FS, 20)
    ylabel('Nondimensional Car Density.', FN, HNB, FS, 18)
    xlabel('Nondimensional distance on road.', FN, HNB, FS, 18)
    hold off
    pause(0.2)
 end
 alpha = alpha_store;
%
if nargout == 0
   clear x_c rho_c c_c t_c
end
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Copyright 1999-2008 Massachusetts Institute of Technology
% Version 01 by Rodolfo R. Rosales  03-25-1999.
% Update     by Rodolfo R. Rosales  02-05-2005.
% Update     by Rodolfo R. Rosales  02-15-2008.
% 
% Permission is hereby granted, without payment, to copy this software
% and its documentation, if any,  for non-profit academic and research
% purposes,  provided that the above copyright notice, this paragraph,
% and the following three paragraphs appear in all copies of this
% software. Use of this software constitutes acceptance of these terms
% and conditions.
%
% IN NO EVENT SHALL MIT, OR THE AUTHOR, BE LIABLE TO ANY PARTY FOR
% DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
% ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
% IF MIT, OR THE AUTHOR, HAVE BEEN ADVISED OF THE POSSIBILITY OF SUCH
% DAMAGE.
%
% MIT, AND THE AUTHOR, SPECIFICALLY DISCLAIM ANY EXPRESS OR IMPLIED
% WARRANTIES INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
% MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, AND
% NON-INFRINGEMENT.
%
% THIS SOFTWARE IS PROVIDED "AS IS." MIT, OR THE AUTHOR, HAVE NO
% OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS,
% OR MODIFICATIONS.
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% EOF
