function [x_c, rho_c, c_c, t_c] = randCFSM(N, rL, rm)
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% function [x_c, rho_c, c_c, t_c] = randCFSM(N, rL, rm)
% ____________________________________________________________________________
% 18.311 (Principles of Applied Mathematics). MIT, Winter 2008. R. R. Rosales.
%
%           Implements the ODE model for traffic flow introduced in the notes:
% "Simplest Car Following Traffic Flow Model".   These are posted in the 18311
% home page. The (nondimensional) model is:
%
%  dx_n/dt = u_n = U(rho_n) for n=1:N,  where rho_n = epsilon/(x_{n+1} - x_n),
%                                              
% x_{n+1} > x_n for all n's  and the density rho_N at the leading car is given
% (equivalently, the velocity u_N of this car is given). Here the car velocity
% function is given by the simple (nondimensional) choice  U(rho) = 4*(1-rho).
% This corresponds to a (nondimensional) flux function  Q(rho) = 4*rho*(1-rho)
% and a characteristic speed c(rho) = (4-8*rho).
%
% The solution is shown via a movie-like slide show. 
% ____________________________________________________________________________
% INPUTS:
% ^^^^^^
%  1) N     = Number of cars in the density disturbance at t = 0 -- see below.
%             Suggestion: 100 < N < 300. Do not take N too large, for then the
%             demo will take a long time to run.     Script will set N = 11 if
%             values outside the range [11, 1000] are prescribed.
%  2) rL    = Nondimensional car density at the leading car:  0.1 < rL < 1.
%             Suggestion: rL around 0.1 to 0.2 works well. If the bounds on rL
%             fail, the script will set rL = 0.1 or rL = 0.9, depending on the 
%             bound violated.
%  3) rm    = Nondimensional minimum car density allowed (see description of
%             initial conditions below).    For rm ouside [0.10, 0.99], script
%             sets rm = 0.1 or rm = 0.9, depending on the bound violated.
%       Note: the reason lower bounds on rL and rm are needed is that small
%             values eventually lead to large computational regions, and these
%             would mean very slow runs.
%
% OUTPUTS:
% ^^^^^^^
% x_c ....... Ntime by N array with the car positions. Each row a time.
% rho_c ..... """"""""""""""""""""""""""""" density.   """"""""""""""""
% c_c ....... """"""""""""""""""""""""" wave speeds.   """"""""""""""""
% t_c ....... Ntime by 1 array with the times.
%
% NOTE: Ntime can be obtained using size(x_c).
%
% Code can be run without any outputs with the command: randCFSM(N, rL, rm);
%
% ____________________________________________________________________________
%
% The initial conditions are taken so that:
%
%      rho_n(0) = equally distributed random number in [rm, 1-rm].
%      x_1(0)   = - pi.
%       x_N(0)  = 0.
%
% See remark 5.1 in the notes for the determination of epsilon.
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%
 Nslide = 100;  %%%%%% THIS IS THE NUMBER OF SLIDES. CAN BE CHANGED.
%
% ###################### Safety checks. ______________________________________
%
 N = floor(N);
 if  N < 11;    N = 11; end
 if  N > 1000;  N = 11; end
 if rm < 0.10; rm = 0.1; end
 if rm > 0.99; rm = 0.9; end
 if rL < 0.10; rL = 0.1; end
 if rL > 1.00; rL = 0.9; end
%
% ###################### Do initial conditions & epsilon. ____________________
%
 r = rm + (1 - rm)*rand(1, N); r(N) = rL;
 c = 4 - 8*r;
 u = 4*(1 - r);
 x = zeros(1, N);
 for n=2:N; x(n) = x(n-1) + 1/r(n-1); end
 epsilon = pi/x(N);
 x = epsilon*x - pi;
 t = 0;
%
% ###################### Compute needed parameters. __________________________
%
 cL  = 4 - 8*rL;
 uL  = 4*(1-rL);
%
% *** Determine how long to compute. We want to do it so there had been plenty
%     of time to get all sorts of shocks to even out the randomness.   The max
%     difference between characteristic speeds gives an idea of how much shock
%     formation potential there is, so we make t_f inversely proportional to
%     this. But we put an absolute maximum to avoid accidental overlarge t_f.
%                  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
 t_f = min(1.5*pi/(max(c) - min(c) + eps), 20);
%
% *** SEE NOTES (section 3): In this case nu=4.  Thus tau_m can be as small as
%                            epsilon/8. For EULER this gives a stability bound
%     dt =< epsilon/4.     Note that this is the same estimate we obtain if we
%                          want to make sure cars do not overlap in one step:
%                                 dt <= (Delta x)/(Delta u) 
%                                     = 0.25*(Delta x)/(Delta rho);
%                          but Delta x = epsilon/rho, with rho and (Delta rho)
%                          up to one.
%     For super extra safety:
%
 dt  = epsilon/15;
%
% *** Determine the time step between slides.
%
 Nstep = Nslide-1;
 dts   = t_f/Nstep;
%
% ###################### Plot initial data. __________________________________
%
 FN = 'FontName'; FS = 'FontSize'; HNB = 'HelveticaNarrowBold';
 LW = 'LineWidth'; MS = 'MarkerSize';
 EJES = [x(1), uL*t_f, min(rm, rL), 1];
%
 figure
 whitebg('k')
 plot(x, r,  '-c', LW, 3)
 hold on
 plot([x(1), x(N)], [r(1), r(N)], '.r', MS, 20)
 plot([x(1), x(N)], [r(1), r(N)], 'og', MS, 10)
 axis(EJES)
 TEXT = num2str(t);
 title(['Nondimensional time t = ', TEXT], FN, HNB, FS, 20)
 ylabel('Nondimensional Car Density.', FN, HNB, FS, 18)
 xlabel('Nondimensional distance on road.', FN, HNB, FS, 18)
 hold off
 pause(0.5)
%
% ###################### Start now calculation. ______________________________
%
 x_c   = x;
 rho_c = r;
 c_c   = c;
 t_c   = t;
 for j=1:Nstep
    tt  = 0;
    while tt < dts
        ddt = ((dts-tt) > dt)*dt + ((dts-tt) <= dt)*(dts-tt);
        x   = x + 4*ddt*(1-r);
        r   = [epsilon./(x(2:N)-x(1:N-1)), rL];
        tt  = tt+ddt;
    end
    t = t+dts;
    c = 4 - 8*r;
    x_c   = [x_c; x];
    rho_c = [rho_c; r];
    c_c   = [c_c; c];
    t_c   = [t_c; t];
 end
%
%
% ###################### Slide show now. _____________________________________
%
 fprintf('\n Ready to start slide show. Press any key.\n')
 pause
 figure
 whitebg('k')
 for n=1:Nslide
    plot(x_c(n, :), rho_c(n, :),  '-c', LW, 3)
    hold on
    plot([x_c(n, 1), x_c(n, N)], [rho_c(n, 1), rho_c(n, N)], '.r', MS, 20)
    plot([x_c(n, 1), x_c(n, N)], [rho_c(n, 1), rho_c(n, N)], 'og', MS, 10)
   axis(EJES)
   TEXT = num2str(t_c(n));
   title(['Nondimensional time t = ', TEXT], FN, HNB, FS, 20)
   ylabel('Nondimensional Car Density.', FN, HNB, FS, 18)
   xlabel('Nondimensional distance on road.', FN, HNB, FS, 18)
   hold off
   pause(0.2)
end
if nargout == 0
   clear x_c rho_c c_c t_c
end
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Copyright 1999-2008 Massachusetts Institute of Technology
% Version 01 by Rodolfo R. Rosales  03-25-1999.
% Update     by Rodolfo R. Rosales  02-05-2005.
% Update     by Rodolfo R. Rosales  02-15-2008.
% 
% Permission is hereby granted, without payment, to copy this software
% and its documentation, if any,  for non-profit academic and research
% purposes,  provided that the above copyright notice, this paragraph,
% and the following three paragraphs appear in all copies of this
% software. Use of this software constitutes acceptance of these terms
% and conditions.
%
% IN NO EVENT SHALL MIT, OR THE AUTHOR, BE LIABLE TO ANY PARTY FOR
% DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
% ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
% IF MIT, OR THE AUTHOR, HAVE BEEN ADVISED OF THE POSSIBILITY OF SUCH
% DAMAGE.
%
% MIT, AND THE AUTHOR, SPECIFICALLY DISCLAIM ANY EXPRESS OR IMPLIED
% WARRANTIES INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
% MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, AND
% NON-INFRINGEMENT.
%
% THIS SOFTWARE IS PROVIDED "AS IS." MIT, OR THE AUTHOR, HAVE NO
% OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS,
% OR MODIFICATIONS.
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% EOF
