function [u0, x, dx] = iniconBDIBS(M, a, b, Par, flag)
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% function [u0, x, dx] = iniconBDIBS(M, a, b, Par, flag)
% ____________________________________________________________________________
% 18.311 (Principles of Applied Mathematics). MIT, Winter 2008. R. R. Rosales.
%
% Generates initial data geared to integration by backdiff_IBS of the Inviscid
% Burgers equation: u_t + (0.5*u^2)_x = 0. That is, it produces: -------------
% dx ...... Space discretization with M intervals: M*dx = b-a.
% x ....... Size M row array with space grid: x = (a+dx):dx:b.
% u0 ...... Initial data: u0 = F(x), with F > 0 and periodic of period (b-a).
% It requires: ---------------------------------------------------------------
% M ....... Number of points in grid.
% a < b ... Ends of period interval.
% Par ..... Vector of parameters to construct F (see below).
% flag .... Integer to choose between several possible F's, as follows:
%
% flag = 1  and Par = [Mean, A1, A2, ... , AN]. Then, with k  = 2*pi/(b-a),
%
%           F(x)=max(eps, Mean+A1*sin(k*x)+A2*sin(2*k*x)+...+AN*sin(N*k*x)).
%
%           That is, F is the positive part of an odd periodic function.
%
% flag = 2  and Par is an array with N entries (N > 1). Then F(x) is piecewise
%           constant on each of a set of equal intervals in [a b],  taking the
%           the value max(eps, Par(n)) in the n_th interval.
%
% flag = 3  and Par = [X1, F1, X2, F2, ... , XN, FN].         Then F(x) is the
%           positive part of a piecewise linear function, taking the values Fn
%           at the point Xn. F vanishes outside the interval [X1, XN].
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Copyright 1999-2008 Massachusetts Institute of Technology
% Version 01 by Rodolfo R. Rosales  03-25-1999.
% Update     by Rodolfo R. Rosales  02-05-2005.
% Update     by Rodolfo R. Rosales  02-10-2008.
% 
% Permission is hereby granted, without payment, to copy this software
% and its documentation, if any,  for non-profit academic and research
% purposes,  provided that the above copyright notice, this paragraph,
% and the following three paragraphs appear in all copies of this
% software. Use of this software constitutes acceptance of these terms
% and conditions.
%
% IN NO EVENT SHALL MIT, OR THE AUTHOR, BE LIABLE TO ANY PARTY FOR
% DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
% ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
% IF MIT, OR THE AUTHOR, HAVE BEEN ADVISED OF THE POSSIBILITY OF SUCH
% DAMAGE.
%
% MIT, AND THE AUTHOR, SPECIFICALLY DISCLAIM ANY EXPRESS OR IMPLIED
% WARRANTIES INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
% MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, AND
% NON-INFRINGEMENT.
%
% THIS SOFTWARE IS PROVIDED "AS IS." MIT, OR THE AUTHOR, HAVE NO
% OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS,
% OR MODIFICATIONS.
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if M <= 0
   error('iniconBDIBS(M, a, b, Par, flag) requires M a positive integer.')
elseif b <= a
   error('iniconBDIBS(M, a, b, Par, flag) requires b > a.')
elseif min(size(Par)) ~= 1
   error('iniconBDIBS(M, a, b, Par, flag) requires Par a non-null VECTOR.')
end
%
M    = ceil(M);
flag = round(flag);   
%
dx = (b - a)/M;
x = (a + dx):dx:b;
%
if flag == 1
   N  = length(Par);
   k  = 2*pi/(b-a);
   u0 = Par(1) * ones(1, M);
   for n = 2:N
       u0 = u0 + Par(n) * sin((n-1) * k * x);
   end
   u0 = max(eps, u0);
elseif flag == 2
    Par = max(eps, Par);
    N   = length(Par);
    dX = (b - a)/N;
    X  = a:dX:b;
    u0 = (Par(1)/2)*(sign(x-a) + sign(X(2)-x));
    for m=2:N
      u0 = u0 + (Par(m)/2)*(sign(x-X(m)) + sign(X(m+1)-x));
    end
elseif flag == 3
    N = length(Par)/2;
    if (N-round(N)) ~= 0
       error('iniconBDIBS requires length(Par) even when flag = 3.')
    end
    X = Par(1:2:2*N); F = Par(2:2:2*N);
    [X, I] = sort(X); F = F(I);
    u0 = zeros(1, M);
    for n=1:N-1
        SLP = (F(n+1)-F(n))/max(eps, X(n+1)-X(n));
        MVL = (F(n)*X(n+1)-F(n+1)*X(n))/max(eps, X(n+1)-X(n));
        u0 = u0 + ((X(n) <= x) & (x < X(n+1))).*(MVL + SLP*x);
    end
        u0 = u0 + (x == X(N)).*(MVL + SLP*x);
    u0 = max(eps, u0);
%
% INSERT OTHER flags here ****************************************************
%
else
   error('Value of flag in iniconBDIBS(M, a, b, Par, flag) not an option.')
end

%
%% EOF
