function u1 = backdiff_IBS(u0, t1, dx)
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% function u1 = backdiff_IBS(u0, t1, dx)
% ____________________________________________________________________________
% 18.311 (Principles of Applied Mathematics). MIT, Winter 2008. R. R. Rosales.
%
% Solves the initial value problem for:   u_t + (0.5 * u^2)_x = 0, 
% with periodic boundary conditions.
%
% A space grid x = x(n) of uniform spacing dx = x(n+1)-x(n) is assumed,  where
% n=1:M and M*dx is the period of the solution in space. Then
%
% uO .... Vector array of length M with initial conditions: u0(n)=u(x(n),0),
% dx .... Space discretization length. Uniform grid assumed in one period.
% t1 .... Time at which the solution is desired. MUST BE t1 > 0.
% u1 .... Vector array of length M with the solution u at time t1.
%
% IMPORTANT: the script uses the simple backward differences scheme:
%
%            u(x, t+dt) = u(x, t) - 0.5*(dt/dx)*(u^2(x, t) - u^2(x-dx, t)),
%
% where dt is the time step.  This is stable ONLY if u is POSITIVE and the CFL
% (*) condition --- dt < dx/(MAX u) --- is satisfied.    The scheme selects dt
% using this condition, but the user MUST provide a u0 that is positive.                             
%
% (*) CFL = Courant, Friedrichs and Lewy. 
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Copyright 1999-2008 Massachusetts Institute of Technology
% Version 01 by Rodolfo R. Rosales  03-25-1999.
% Update     by Rodolfo R. Rosales  02-05-2005.
% Update     by Rodolfo R. Rosales  02-10-2008.
% 
% Permission is hereby granted, without payment, to copy this software
% and its documentation, if any,  for non-profit academic and research
% purposes,  provided that the above copyright notice, this paragraph,
% and the following three paragraphs appear in all copies of this
% software. Use of this software constitutes acceptance of these terms
% and conditions.
%
% IN NO EVENT SHALL MIT, OR THE AUTHOR, BE LIABLE TO ANY PARTY FOR
% DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
% ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
% IF MIT, OR THE AUTHOR, HAVE BEEN ADVISED OF THE POSSIBILITY OF SUCH
% DAMAGE.
%
% MIT, AND THE AUTHOR, SPECIFICALLY DISCLAIM ANY EXPRESS OR IMPLIED
% WARRANTIES INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
% MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, AND
% NON-INFRINGEMENT.
%
% THIS SOFTWARE IS PROVIDED "AS IS." MIT, OR THE AUTHOR, HAVE NO
% OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS,
% OR MODIFICATIONS.
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
                            
if min(size(u0)) ~= 1
   error('backdiff_IBS(u0, t1, dx) requires u0 to be a non-null vector.')
elseif t1 <= 0
   error('backdiff_IBS(u0, t1, dx) requires t1 > 0.')
elseif dx <= 0
   error('backdiff_IBS(u0, t1, dx) requires dx > 0.')
elseif min(u0) <= 0
   error('backdiff_IBS(u0, t1, dx) requires u0 > 0.')
end
%
M  = length(u0);
dt = min(0.95*dx/max(u0), t1);
Nt = ceil(t1/dt);
dt = t1/Nt;
D  = 0.5*dt/dx;
u1 = u0;
%
for n=1:Nt
   u1 = u1 - D * (u1.^2 - u1([M, 1:M-1]).^2);
end
%
%% END OF FILE.
