function OUTPUT = GBNS_lecture_FIG
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% function OUTPUT = GBNS_lecture_FIG 
%          Creates the control console and figure used by GNBS_lecture
%
% 18.311 (Principles of Applied Mathematics). MIT, Winter 2008. R. R. Rosales.
% ____________________________________________________________________________
%
% OUTPUT is a structure with the following fields (HANDLES):
%
% OUTPUT.fgN ..... Figure number.
% OUTPUT.Hax ..... Axes handle.
% OUTPUT.Hbf ..... Right Frame, Top (buttons) handle.
% OUTPUT.Hbt ..... Array with button handles  (order bottom to top.)
% OUTPUT.Hdp ..... Array with display handles (order bottom to top.)
% OUTPUT.Hdf ..... Right Frame, Bottom (displays) handle.
% OUTPUT.Hsf ..... Slider frame handle.
% OUTPUT.Hsl ..... Slider handle.
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Copyright 2001-2008 Massachusetts Institute of Technology
% Version 01 by Rodolfo R. Rosales  03-12-2001.
% Update     by Rodolfo R. Rosales  01-20-2005.
% Update     by Rodolfo R. Rosales  02-20-2008.
% 
% Permission is hereby granted, without payment, to copy this software
% and its documentation, if any,  for non-profit academic and research
% purposes,  provided that the above copyright notice, this paragraph,
% and the following three paragraphs appear in all copies of this
% software. Use of this software constitutes acceptance of these terms
% and conditions.
%
% IN NO EVENT SHALL MIT, OR THE AUTHOR, BE LIABLE TO ANY PARTY FOR
% DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
% ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
% IF MIT, OR THE AUTHOR, HAVE BEEN ADVISED OF THE POSSIBILITY OF SUCH
% DAMAGE.
%
% MIT, AND THE AUTHOR, SPECIFICALLY DISCLAIM ANY EXPRESS OR IMPLIED
% WARRANTIES INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
% MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, AND
% NON-INFRINGEMENT.
%
% THIS SOFTWARE IS PROVIDED "AS IS." MIT, OR THE AUTHOR, HAVE NO
% OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS,
% OR MODIFICATIONS.
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Fsz = 0.80; % Vertical dimension of the figure,  as a fraction of
%           % the smallest of the width and length of the screen.
% ****************************************************************************
% General Window Structure Parameters: _______________________________________
%
% ----------------------------------------------------------------
% WINDOW DESCRIPTION AND SELECTABLE PARAMETERS: ==================
% --- Plotting area is square, on the top-left side of the figure.
% --- On the right, a vertical frame with buttons and displays ---
%     actually, two frames: top for buttons; bottom for displays.
% --- On the bottom there is an horizontal frame with a slider. It
%     has the same width as the plotting area.
% These objects are separated from the figure edges and each other
% by various distances.    The parameters below characterize these
% distances, and the sizes of the various objects. The lengths are
% given in AXIS UNITS (in them the plotting area is 1 by 1.)
%
pa1 = 0.160; % Thickness of space around plotting area: left.
pa2 = 0.150; % Thickness of space around plotting area: bottom.
pa3 = 0.150; % Thickness of space around plotting area: right.
pa4 = 0.150; % Thickness of space around plotting area: top.
%
%            % Note that what would be m1 below is equal to pa1.
ma2 = 0.020; % Margin from frames to figure edge: bottom.
ma3 = 0.020; % Margin from frames to figure edge: right.
ma4 = 0.040; % Margin from frames to figure edge: top.
%
sfh = 0.090; % Slider frame height.
bfw = 0.350; % Button's frame width.
bfd = 0.400; % Button's frame height fraction used by displays.
%
bwf = 0.90;  % Fraction of frame width  taken up by the buttons.
bhf = 0.85;  % Fraction of frame height taken up by the buttons.
dwf = 0.87;  % Fraction of frame width  taken up by the displays.
dhf = 0.70;  % Fraction of frame height taken up by the displays.
%
% ----------------------------------------------------------------
% BUTTON INPUTS (index goes bottom to top): ======================
% NOTE:
% --- B_Color: Background Color.
% --- F_Color: Foreground Color (text.)
% --- R_Height (Relative Height): Weight used for button height. 
%
n = 1;
Button{n}.CallBack = 'GBNS_lecture(''Close'')';
Button{n}.String   = 'Close';
Button{n}.B_Color  = 'r';
Button{n}.F_Color  = 'y';
Button{n}.FontSize =  0.50;
Button{n}.Enable   = 'on';
Button{n}.Style    = 'pushbutton';
Button{n}.R_Height =  1.00;
n = n+1;
Button{n}.CallBack = 'GBNS_lecture(''Part 3'')';
Button{n}.String   = 'Part 3';
Button{n}.B_Color  = 'g';
Button{n}.F_Color  = 'k';
Button{n}.FontSize =  0.50;
Button{n}.Enable   = 'on';
Button{n}.Style    = 'pushbutton';
Button{n}.R_Height =  1.00;
n = n+1;
Button{n}.CallBack = 'GBNS_lecture(''Part 2'')';
Button{n}.String   = 'Part 2';
Button{n}.B_Color  = 'g';
Button{n}.F_Color  = 'k';
Button{n}.FontSize =  0.50;
Button{n}.Enable   = 'on';
Button{n}.Style    = 'pushbutton';
Button{n}.R_Height =  1.00;
n = n+1;
Button{n}.CallBack = 'GBNS_lecture(''Part 1'')';
Button{n}.String   = 'Part 1';
Button{n}.B_Color  = 'g';
Button{n}.F_Color  = 'k';
Button{n}.FontSize =  0.50;
Button{n}.Enable   = 'on';
Button{n}.Style    = 'pushbutton';
Button{n}.R_Height =  1.00;
n = n+1;
Button{n}.CallBack = 'GBNS_lecture(''Reset View'')';
Button{n}.String   = 'Reset View';
Button{n}.B_Color  = 'c';
Button{n}.F_Color  = 'b';
Button{n}.FontSize =  0.45;
Button{n}.Enable   = 'off';
Button{n}.Style    = 'pushbutton';
Button{n}.R_Height =  0.60;
n = n+1;
Button{n}.CallBack = 'GBNS_lecture(''Rotate'')';
Button{n}.String   = 'Rotate';
Button{n}.B_Color  = 'r';
Button{n}.F_Color  = 'y';
Button{n}.FontSize =  0.50;
Button{n}.Enable   = 'off';
Button{n}.Style    = 'pushbutton';
Button{n}.R_Height =  1.00;
n = n+1;
Button{n}.CallBack = 'GBNS_lecture(''Help'')';
Button{n}.String   = 'Help';
Button{n}.B_Color  = 'y';
Button{n}.F_Color  = 'r';
Button{n}.FontSize =  0.50;
Button{n}.Enable   = 'on';
Button{n}.Style    = 'pushbutton';
Button{n}.R_Height =  1.00;
%
Nbt = n;% Number of buttons.
%
% ----------------------------------------------------------------
% SLIDER INPUTS. =================================================
%
Slider.Value    = 3;
Slider.Min      = 1;
Slider.Max      = 6;
Slider.Step     = [0.2 0.4];
Slider.Enable   = 'off';
Slider.CallBack = '';
%
% ----------------------------------------------------------------
% Display INPUTS (index goes bottom to top): =====================
% NOTE:
% --- B_Color: Background Color.
% --- F_Color: Foreground Color (text.)
% --- R_Height (Relative Height): Weight used for display height. 
%
n = 1;
Display{n}.String   = ' ';
Display{n}.B_Color  = 'w';
Display{n}.F_Color  = 'b';
Display{n}.FontSize =  0.55;
Display{n}.Visible  = 'off';
Display{n}.Style    = 'Text';
Display{n}.R_Height =  1.00;
n = n+1;
Display{n}.String   = ' ';
Display{n}.B_Color  = 'w';
Display{n}.F_Color  = 'b';
Display{n}.FontSize =  0.55;
Display{n}.Visible  = 'off';
Display{n}.Style    = 'Text';
Display{n}.R_Height =  1.00;
n = n+1;
Display{n}.String   = ' ';
Display{n}.B_Color  = 'w';
Display{n}.F_Color  = 'b';
Display{n}.FontSize =  0.55;
Display{n}.Visible  = 'off';
Display{n}.Style    = 'Text';
Display{n}.R_Height =  1.00;
n = n+1;
Display{n}.String   = ' ';
Display{n}.B_Color  = 'w';
Display{n}.F_Color  = 'b';
Display{n}.FontSize =  0.55;
Display{n}.Visible  = 'off';
Display{n}.Style    = 'Text';
Display{n}.R_Height =  1.00;
n = n+1;
Display{n}.String   = ' ';
Display{n}.B_Color  = 'w';
Display{n}.F_Color  = 'b';
Display{n}.FontSize =  0.55;
Display{n}.Visible  = 'off';
Display{n}.Style    = 'Text';
Display{n}.R_Height =  1.00;
n = n+1;
Display{n}.String   = ' ';
Display{n}.B_Color  = 'w';
Display{n}.F_Color  = 'b';
Display{n}.FontSize =  0.55;
Display{n}.Visible  = 'off';
Display{n}.Style    = 'Text';
Display{n}.R_Height =  1.00;
n = n+1;
Display{n}.String   = ' ';
Display{n}.B_Color  = 'w';
Display{n}.F_Color  = 'b';
Display{n}.FontSize =  0.55;
Display{n}.Visible  = 'off';
Display{n}.Style    = 'Text';
Display{n}.R_Height =  1.00;
%
Ndp = n;% Number of displays.
%
% ----------------------------------------------------------------
% CALCULATE NOW OTHER PARAMETERS: ================================
% The axis and buttons, in the figure, are placed using normalized
% units (measure everything in terms of the figure dimensions, the
% figure is then 1 by 1.)    To convert from these FIGURE UNITS to
% AXIS UNITS (plotting area is 1 by 1),  we use the factors below,
% that measure the figure dimensions in axis units.
%
fgw = pa1 + 1.0 + pa3 + bfw + ma3; % Figure width;  AXIS UNITS.
fgh = ma2 + sfh + pa2 + 1.0 + pa4; % Figure height; AXIS UNITS.
figTOaxis = [fgw, fgh, fgw, fgh];  % Conversion array.
%
% Then the position of the axes for plotting, and the frames, are:
% (FIGURE UNITS.)
%
PosAxes = [pa1, ma2+sfh+pa2, 1.0, 1.0]./figTOaxis;
        % Position of the axes.
PosFsld = [pa1, ma2, 1.0, sfh]./figTOaxis;
        % Position of the frame on the bottom (for slider.)
PosFRdp = [pa1+1.0+pa3, ma2, bfw, bfd*(fgh-ma2-ma4)]./figTOaxis;
        % Position of the right frame --- displays section.
PosFRbt = [pa1+1.0+pa3, ma2+bfd*(fgh-ma2-ma4), bfw, ...
          (1-bfd)*(fgh-ma2-ma4)]./figTOaxis;
        % Position of the right frame --- buttons section.
%
% ****************************************************************************
% Setup the figure and buttons. ______________________________________________
%
fgN = figure('Name',        ' PDE numerical scheme stability. ', ...
             'NumberTitle', 'off', ...
             'Visible',     'off');
OUTPUT.fgN = fgN;
% ----------------------------------------------------------------
% --- Swd = Screen Width  \ SMD = Screen Minimum Dimension. ======
%     Sht = Screen Height /
%
Ssz = get(0, 'ScreenSize');
Swd = Ssz(3);
Sht = Ssz(4);
SMD = min(Swd, Sht);
%
PosFig = [Swd, Sht, 0, 0] + ...
         Fsz*SMD*[ -1.05, -1.08*fgh/fgw, 1, fgh/fgw];
set(fgN, 'Position', PosFig)
colordef(fgN, 'black')
% ----------------------------------------------------------------
% --- Axes. ======================================================
%
Hax = axes('Units',    'normalized', ...
           'Position',  PosAxes, ...
           'Visible',  'off');
OUTPUT.Hax = Hax;
% ----------------------------------------------------------------
% --- Right Frame, Top (buttons). ================================
%
Hbf = uicontrol('Style',           'frame', ...
                'Units',           'normalized', ...
                'Position',         PosFRbt, ...
                'BackgroundColor',  [0.00 0.50 0.50]);
OUTPUT.Hbf = Hbf;
% ----------------------------------------------------------------
% --- Create buttons (Order bottom to top.) ======================
%
% Button parameters.
% Width and height in .................. AXIS  UNITS.
% Margin and vertical separations in ... FIGURE UNITS
% Position in .......................... FIGURE UNITS.
%
bwd = bwf*bfw;                            % Width.
bht = PosFRbt(4)*fgh*bhf/Nbt;             % Average height.
bmg = (bfw-bwd)/(2*fgw);                  % Margin from frame.
bsp = (PosFRbt(4) - Nbt*bht/fgh)/(Nbt+1); % Vertical separation.
%
% PosButt is the position array for the buttons.
%    It is updated in the "for" loop for each button.
% Temp1 here has total "weight" for button heights.
% Temp2 is used to get each individual button height.
%
Temp1 = 0;  for n=1:Nbt;  Temp1 = Temp1+Button{n}.R_Height;  end
PosButt = [PosFRbt(1)+bmg, PosFRbt(2)+bsp, bwd/fgw, 0];
%
for n=1:Nbt
   Temp2      = (bht/fgh)*(Nbt/Temp1)*Button{n}.R_Height;
   PosButt(4) = Temp2;
   Hbt(n) = uicontrol('Style',            Button{n}.Style, ...
                      'Units',           'normalized', ...
                      'Position',         PosButt, ...
                      'String',           Button{n}.String, ...
                      'Callback',         Button{n}.CallBack, ...
                      'BackgroundColor',  Button{n}.B_Color, ...
                      'ForegroundColor',  Button{n}.F_Color, ...
                      'Enable',           Button{n}.Enable, ...
                      'FontWeight',      'Bold', ...
                      'FontUnits',       'Normalized', ...
                      'FontSize',         Button{n}.FontSize);
   PosButt = PosButt + (bsp + Temp2)*[0 1 0 0];
end
OUTPUT.Hbt = Hbt;
% ----------------------------------------------------------------
% --- Right Frame, Bottom (displays). ============================
%
Hdf = uicontrol('Style',           'frame', ...
                'Units',           'normalized', ...
                'Position',         PosFRdp, ...
                'BackgroundColor',  [0.00 0.00 1.00]);
OUTPUT.Hdf = Hdf;
% ----------------------------------------------------------------
% --- Create displays (Order bottom to top.) =====================
%
% Display parameters.
% Width and height in .................. AXIS  UNITS.
% Margin and vertical separations in ... FIGURE UNITS
% Position in .......................... FIGURE UNITS.
%
dwd = dwf*bfw;                            % Width.
dht = PosFRdp(4)*fgh*dhf/Ndp;             % Average height.
dmg = (bfw-dwd)/(2*fgw);                  % Margin from frame.
dsp = (PosFRdp(4) - Ndp*dht/fgh)/(Ndp+1); % Vertical separation.
%
% PosDisp is the position array for the displays.
%    It is updated in the "for" loop for each display.
% Temp1 here has total "weight" for display heights.
% Temp2 is used to get each individual display height.
%
Temp1 = 0;  for n=1:Ndp;  Temp1 = Temp1+Display{n}.R_Height;  end
PosDisp = [PosFRdp(1)+dmg, PosFRdp(2)+dsp, dwd/fgw, 0];
%
for n=1:Ndp
   Temp2      = (dht/fgh)*(Ndp/Temp1)*Display{n}.R_Height;
   PosDisp(4) = Temp2;
   Hdp(n) = uicontrol('Style',            Display{n}.Style, ...
                      'Units',           'normalized', ...
                      'Position',         PosDisp, ...
                      'String',           Display{n}.String, ...
                      'BackgroundColor',  Display{n}.B_Color, ...
                      'ForegroundColor',  Display{n}.F_Color, ...
                      'Visible',          Display{n}.Visible, ...
                      'FontWeight',      'Bold', ...
                      'FontUnits',       'Normalized', ...
                      'FontSize',         Display{n}.FontSize);
   PosDisp = PosDisp + (dsp + Temp2)*[0 1 0 0];
end
OUTPUT.Hdp = Hdp;
% ----------------------------------------------------------------
% --- Slider Frame. ==============================================
%
Hsf = uicontrol('Style',           'frame', ...
                'Units',           'normalized', ...
                'Position',         PosFsld, ...
                'BackgroundColor',  [0.00 0.50 0.50]);
OUTPUT.Hsf = Hsf;
% ----------------------------------------------------------------
% --- Create slider. =============================================
%
PosSldr = [PosFsld(1)+0.05*PosFsld(3), ...
           PosFsld(2)+0.05*PosFsld(4), ...
           0.90*PosFsld(3), 0.90*PosFsld(4)];
Hsl = uicontrol('Style',           'slider', ...
                'Value',            Slider.Value, ...
                'Min',              Slider.Min, ...
                'Max',              Slider.Max, ...
                'SliderStep',       Slider.Step, ...
                'Units',           'normalized', ...
                'Position',         PosSldr, ...
                'Enable',           Slider.Enable, ...
                'CallBack',         Slider.CallBack);   
%
OUTPUT.Hsl = Hsl;
%
% ****************************************************************************
% Set startup message. _______________________________________________________
%
set(fgN, 'Visible', 'on');
%
global global_GBNS_lect_FigPar
global_GBNS_lect_FigPar.Hax = Hax;
%
GBNS_lecture_MSG1
%
% ****************************************************************************
% Delete output if not needed. _______________________________________________
%
if nargout == 0
   clear OUTPUT
end
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%% EOF
