function [x, T, U, V, M] = GBNS_GScheme(u0, v0, t_f, a, b, Nframes)
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% function [x, T, U, V, M] = GBNS_GScheme(u0, v0, t_f, a, b, Nframes)
%
% 18.311 (Principles of Applied Mathematics). MIT, Winter 2008. R. R. Rosales.
% ____________________________________________________________________________
%
% Solves the WAVE EQUATION , written in the form:    u_t = v  and  v_t = u_xx,
% for the time interval 0 <= t <= t_f.
% Uses PERIODIC BOUNDARY CONDITIONS for x in [a, b]:   u(a, t) = u(b, t),
%                                                      v(a, t) = v(b, t),
% and initial values given by the ROW arrays u0 and v0.
%   
% Algorithm: _________________________________________________________________
%
% u(x,t+dt) = u(x,t) +  v(x,t)*dt + u2(x,t)*eta,
% v(x,t+dt) = v(x,t) + u2(x,t)*nu + v2(x,t)*eta,
%
% where nu  = dt/(dx)^2,
%       eta = (dt/dx)^2,
%       u2 = u(x+dx, t) - 2 u(x, t) + u(x-dx, t),
% and   v2 = v(x+dx, t) - 2 v(x, t) + v(x-dx, t).  ##### SEE readmeGBNS. #####
%
% VARIABLES: _________________________________________________________________
%
% u0 ........ Initial values for u. Row array. \ N = length(u0) = length(v0).
% v0 ........ Initial values for v. Row array. / 
% t_f ....... Final time of computation. Solution computed for 0 <= t <= t_f.
% a ......... Left  end of integration interval.
% b ......... Right end of integration interval.
% Nframes ... OPTIONAL. An integer >= 2.
%             IF AN INPUT. On output solution given for the Nframes times:
%                                  T = (0:DT:t_f)',
%             where               DT = t_f/(Nframes-1),
%             and                  M = Nframes.
%
%             NOTE THAT the ACTUAL TIME STEP (i.e. dt) used in the computation
%             is selected using the conditions
%             --- dt =< 0.5*dx, where dx = (b-a)/N is the space mesh size.
%             --- t_f/dt is an integer, with dt as large as possible.
%
%             Thus:           dt = t_f/ceil(2*t_f/dx),
%
%             with the solution known/computed for the
%
%                           Mnum = 1+ceil(N*t_f)
%             times         Tnum = (0:dt:t_f)'.
%
%             The solution at the requested times is computed by doing a small
%             time step from the closest value in Tnum below each time in T.
%
%             IF Nframes IS NOT AN INPUT, on output: M = Mnum and T = Tnum.
%
% x ......... Space grid. Row array  x = a:dx:b-dx, with dx = (b-a)/N.
% T ......... Column array of length M with the computed times;   T(1) = 0 and
%             T(M) = t_f. The times are equally spaced.
% U ......... MxN array with solution u. Row n corresponds to u at time T(n).
% V ......... MxN array with solution v. Row n corresponds to v at time T(n).
% M ......... Number of times in output (including the initial value t = 0).
%
% VISUALIZATION: _____________________________________________________________
%                To visualize the solution can:
% 1) Use mesh(x,T,U) or mesh(x,T,V) for 3-D plots.
% 2) Use surf(x,T,U) or surf(x,T,V) for 3-D plots.
% 3) Use plot(x, U(n,:) or plot(x, V(n,:) for individual time slices.
%
% ##### For more information see:  readmeGBNS.
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% ------------------------------------------------------------- Safety checks.
%
N = length(u0);
%
if t_f < 100*eps;  t_f = 100*eps;  end
%
if length(v0) ~= N
   error('u0 and v0 must have the same length.')
end
%
% ------------------------- Define parameters needed for the numerical scheme.
%
x   = a + (b-a)*(0:N-1)/N;
dx  = (b-a)/N;
dt  = t_f/ceil(2*t_f/dx);
eta = (dt/dx)^2;
nu  = dt/dx^2;
%
if nargin < 6
   M = 1 + ceil(t_f/dt);
else
   M = max(2, ceil(Nframes));
end
%
T = (t_f/(M-1))*((0:M-1)');
%
% --------------------------------------------------------------- Initialize.
%
t = 0;   u = u0;   v = v0;   U = u;   V = v;
%
% --------------------------------------------------------------- Now compute.
%
for n=2:M
   while (t + dt*1.01) < T(n)
      u2 = u([2:N, 1]) - 2*u + u([N, 1:(N-1)]);
      v2 = v([2:N, 1]) - 2*v + v([N, 1:(N-1)]);
      u  = u + dt*v  + eta*u2;
      v  = v + nu*u2 + eta*v2;
      t  = t + dt;
   end
   DT = T(n) - t;
   u2 = u([2:N, 1]) - 2*u + u([N, 1:(N-1)]);
   v2 = v([2:N, 1]) - 2*v + v([N, 1:(N-1)]);
   u  = u + DT*v          + (DT^2/dt^2)*eta*u2;
   v  = v + (DT/dt)*nu*u2 + (DT^2/dt^2)*eta*v2;
   U  = [U; u];
   V  = [V; v];
   t  = T(n);
end
%
if nargout == 0
   clear
end
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Copyright 2001-2008 Massachusetts Institute of Technology
% Version 01 by Rodolfo R. Rosales  02-14-2001.
% Update     by Rodolfo R. Rosales  02-25-2008.
% 
% Permission is hereby granted, without payment, to copy this software
% and its documentation, if any,  for non-profit academic and research
% purposes,  provided that the above copyright notice, this paragraph,
% and the following three paragraphs appear in all copies of this
% software. Use of this software constitutes acceptance of these terms
% and conditions.
%
% IN NO EVENT SHALL MIT, OR THE AUTHOR, BE LIABLE TO ANY PARTY FOR
% DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
% ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
% IF MIT, OR THE AUTHOR, HAVE BEEN ADVISED OF THE POSSIBILITY OF SUCH
% DAMAGE.
%
% MIT, AND THE AUTHOR, SPECIFICALLY DISCLAIM ANY EXPRESS OR IMPLIED
% WARRANTIES INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
% MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, AND
% NON-INFRINGEMENT.
%
% THIS SOFTWARE IS PROVIDED "AS IS." MIT, OR THE AUTHOR, HAVE NO
% OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS,
% OR MODIFICATIONS.
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%% EOF
