function [x_c, rho_c, c_c, t_c] = quadCFSM(Nh, rho_N)
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% function [x_c, rho_c, c_c, t_c] = quadCFSM(Nh, rho_N)
% ____________________________________________________________________________
% 18.311 (Principles of Applied Mathematics). MIT, Winter 2008. R. R. Rosales.
%
%           Implements the ODE model for traffic flow introduced in the notes:
% "Simplest Car Following Traffic Flow Model".   These are posted in the 18311
% home page. The (nondimensional) model is:
%
%                    dx_n/dt = u_n = U(rho_n)   for n=1:N,
%
% where 0 < rho_n = epsilon/(x_(n+1) - x_n)) < 1  and the density rho_N at the
% leading car is given  (equivalently, the velocity u_N of this car is given).
% The car velocity function U is given by the simple (nondimensional) choice
%
%   U(rho) = 4*(1-rho)  ===> Q(rho) = 4*rho*(1-rho) and c(rho) = (4-8*rho).
% 
% SEE THE NOTES, (EXAMPLE 4.1 and section 5) for more details.  In particular,
% how the initial conditions are selected, etc.
%
% THE SOLUTION IS SHOWN VIA A MOVIE-LIKE SLIDE SHOW.
% ____________________________________________________________________________
% Description of the outputs and required inputs:
% ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
% INPUTS
%  1) Nh    = Number of cars in the density hump disturbance at t = 0.
%             Suggestion: 100 < Nh < 300.   Do not take Nh too large, for then
%             the script will take a long time to run.    If Nh is outside the
%             range [11, 1500], the script sets Nh = 11.
%             The actual number of cars used (i.e.: N) is larger than Nh.
%  2) rho_N = Nondimensional car density at the first car:  0 < rho_N < 1.
%             Suggestion: rho_N around 0.2 works well.  On the other hand, the 
%             script does not work too well outside [0.1, 0.8].  If values not
%             in this range are entered, the script uses rho_N = 0.5.
%
% OUTPUTS
% x_c ....... Ntime by N array with the car positions. Each row a time.
% rho_c ..... """"""""""""""""""""""""""""" density.   """"""""""""""""
% c_c ....... """"""""""""""""""""""""" wave speeds.   """"""""""""""""
% t_c ....... Ntime by 1 array with the times.
%
% NOTE: N and Ntime can be obtained using size(x_c).
%
% Code can be run without any outputs with the command: quadCFSM(Nh, rho_N);
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%
 Ntime = 100;  %%%%%% THIS IS THE NUMBER OF SLIDES. CAN BE CHANGED.
 ratio = 6;    %%%%%% ratio = dist(shock, corner)/dist(corner, leading car)
%              %%%%%% when calculation ends. CAN BE CHANGED.
 rN = rho_N;
%
% ###################### Safety checks. ______________________________________
%
 Nh = floor(Nh);
 if Nh <   11; Nh =  11; end
 if Nh > 1500; Nh =  11; end
 if rN <  0.1; rN = 0.5; end
 if rN >  0.8; rN = 0.5; end
%
% ###################### Do now hump (sine of car number) & epsilon. _________
%
 xh = (0:1:(Nh-1))*pi/(Nh-1);   
 rh = rN + (1-rN)*sin(xh);
 ch = 4 - 8*rh;
 for n=2:Nh; xh(n) = xh(n-1) + 1/rh(n-1); end
 epsilon = pi/xh(Nh);
 xh = epsilon*xh - pi;
%
% ###################### Compute needed parameters. __________________________
%
 cN  = 4 - 8*rN;% ------------------ Leading car characteristic speed.
 uN  = 4*(1-rN);% ------------------ Leading car velocity.
% ---------------------------------- Stuff needed to approximate the solution.
%                                    by equation (5.7). See the NOTES.
 A   = 8*(epsilon*(Nh-1) - pi*rN);%  Formula below equation (5.5)).
% ---------------------------------- Nl is (basically) the notes \ell.    Note
%                                    that rh(Nl) is halfway up the rh profile:
%                                    rh(Nl) is about (rN+1)/2.
 Nl  = ceil(5*Nh/6);
% ---------------------------------- For be we use equation (5.8)
 B   = 16*(epsilon*(Nh-Nl) + xh(Nl)*rN)/xh(Nl)^2;
 B   = (B <= 0)*1e-8 + (B > 0)*B;% - Some safety here.
% ---------------------------------- Use (5.7) for the shock path.   Use it to
%                                    set final a time t_f so that at t = t_f:
% 
%              ratio >= dist(shock-corner)/dist(corner-leading car),
%
%                                    where ratio is defined at the script top.
%                                    This will make the corner at cN*t visible
%                                    separating it from the lead car at uN*t.
 t_f = A/(ratio*(uN-cN))^2;
 t_f = t_f*(1 + sqrt(1+2/B/t_f));
% ---------------------------------- Make sure hump trailing has overtaken the
%                                    top ===> well formed shock.
 t_f = max(t_f, pi/2/(cN+4));
% ---------------------------------- Make  sure shock has advanced to have the
%                                    value rh(Nl) about to be swallowed.   Use
%                                    (5.7)  and  equate rh(Nl) = rho_S.                                   
 t_f = max(t_f, 2*A/(8*(rh(Nl)-rN))^2 - 1/B);
% ---------------------------------- Slope of the wave speed at t=0 on hump.
 Sh = [(ch(2:Nh)-ch(1:Nh-1)).*rh(1:Nh-1)/epsilon, 0];
% ---------------------------------- Shock formation time, etc.
 [S_m, n_m] = min(Sh);
 z_m = xh(n_m);
 r_m = rh(n_m);
 c_m = ch(n_m);
 t_S = -1/S_m;% -------------------- Shock formation time.
 x_S = z_m + c_m*t_S;% ------------- Shock formation location.
% ---------------------------------- Trailing car at xT on t = 0 reaches shock
%                                    at t = t_f.   We add safety and stop just
%                                    short of trailing car at shock.
 xT  = (cN-uN)*t_f - sqrt(2*A*(1+B*t_f)/B);
 xT  = min(xT, -pi) - 3*epsilon/rN;
% ---------------------------------- SEE NOTES (section 3):  Here nu = 4. Thus
%                                    tau_m can be as small as epsilon/8.   For
% EULER this gives a stability bound dt =< epsilon/4.  This is the same as the
% bound to make sure cars do not overlap in one step:  dt<=(Delta x)/(Delta u) 
% = 0.25*(Delta x)/(Delta rho) --- but Delta x = epsilon/rho,     with rho and
% (Delta rho) up to one. Just for extra safety:
%
 dt  = epsilon/5;
% ---------------------------------- Time step between slides.
 Nstep = Ntime-1;
 dts   = t_f/Nstep;
%
% ###################### Construct tail behind bump. _________________________
%
 dxt = epsilon/rN;
 Nt  = ceil(-(xT+pi)/dxt);
 xt = (-pi-Nt*dxt):dxt:(-pi-dxt);
 rt = rN*ones(1, Nt);
%
% ###################### Do now the initial data. ____________________________
%
 t = 0;
 x = [xt, xh];
 r = [rt, rh];
 N = Nt + Nh;
 c = 4 - 8*r;
%
% ###################### Start now calculation. ______________________________
%
 x_c   = x;
 rho_c = r;
 c_c   = c;
 t_c   = t;
%
 FN   = 'FontName';
 FS   = 'FontSize';
 HNB  = 'HelveticaNarrowBold';
 LW   = 'LineWidth';
 MS   = 'MarkerSize';
 EJES = [x(1), uN*t_f, max(rN-0.1, 0), 1];
% ---------------------------------- Plot initial values.
figure
whitebg('k')
plot(x, r,  '-c', LW, 3)
hold on
plot([x(1), x(N)], [r(1), r(N)], '.r', MS, 20) % Red dot + green circle
plot([x(1), x(N)], [r(1), r(N)], 'og', MS, 10) % on first and last car.
axis(EJES)
TEXT = num2str(t);
title(['Nondimensional time t = ', TEXT], FN, HNB, FS, 20)
ylabel('Nondimensional Car Density.', FN, HNB, FS, 18)
xlabel('Nondimensional distance on road.', FN, HNB, FS, 18)
hold off
%
fprintf('\n Calculation proceeding. Wait while slides are made.\n')
pause(0.5)
%
for j=1:Nstep
   tt  = 0;
   while tt < dts
       ddt = ((dts-tt) > dt)*dt + ((dts-tt) <= dt)*(dts-tt);
       x   = x + 4*ddt*(1-r);
       r   = [epsilon./(x(2:N)-x(1:N-1)), rN];
       tt  = tt+ddt;
   end
   t = t+dts;
   c = 4 - 8*r;
   x_c   = [x_c; x];
   rho_c = [rho_c; r];
   c_c   = [c_c; c];
   t_c   = [t_c; t];
end
%
% ###################### Calculate shock path and corner path. _______________
%   
%     Shock path needed only for t > t_S,    while x_slope and r_slope are
%     needed only for t < t_f (slope of rho along blow up characteristic).
%     Note safety to prevent a possible infinity when t = t_S.
%
 x_corn1 = cN*t_c;% ----------------------------- Leading  corner.
 x_corn2 = x_corn1-pi;% ------------------------- Trailing corner.
 x_shock = x_corn1 - sqrt(2*A*(1+B*t_c)/B);% ---- Shock path.
 r_shock = rN + (1/8)*sqrt(2*A*B./(1+B*t_c));% -- Value of rho at shock. 
 x_slope = z_m + c_m*t_c;% ---------------------- Breaking characteristic.
 r_slope = -0.125*S_m./(abs(1+S_m*t_c)+eps);% --- Slope of rho at x_slope.
%
fprintf('\n ______________________________________________________________')
fprintf('\n Leading and trailing cars marked by a GREEN ENCIRCLED RED DOT.')
fprintf('\n RED LINE with CROSS: shock position and value of rho predicted')
fprintf('\n                      by continuum theory.')
fprintf('\n BLUE LINE and GREEN ENCIRCLED YELLOW DOT: indicate the place &')
fprintf('\n                      slope along the characteristic where  the')
fprintf('\n                      continuum theory predics first breakdown.')
fprintf('\n The MAGENTA LINES mark the predicted positions of corners.')
fprintf('\n ______________________________________________________________\n')
fprintf('\n READY TO START SLIDE SHOW. Press any key.\n')
fprintf('\n')
%
pause
figure
whitebg('k')
for n=1:Ntime
   plot(x_c(n, :), rho_c(n, :),  '-c', LW, 3)
   hold on
   plot([x_c(n, 1), x_c(n, N)], [rho_c(n, 1), rho_c(n, N)], '.r', MS, 20)
   plot([x_c(n, 1), x_c(n, N)], [rho_c(n, 1), rho_c(n, N)], 'og', MS, 10)
   if t_c(n) <= t_S
      Y = r_m + (-1:1)*(1-rN)/5;
      X = x_slope(n) + (Y-r_m)/r_slope(n);
      plot(X, Y, '-b', LW, 2)
      plot(X(2), Y(2), '.y', MS, 20)
      plot(X(2), Y(2), 'og', MS, 10)
   elseif t_c(n) > t_S
      plot(x_shock(n), r_shock(n), '+r', MS, 12)
      plot([x_shock(n), x_shock(n)], [EJES(3), r_shock(n)], '-r', LW, 2)
   end
   if x_corn2(n) < x_shock(n)
      plot([x_corn2(n), x_corn2(n)], [EJES(3), rN], '-m', LW, 2)
   end
   plot([x_corn1(n), x_corn1(n)], [EJES(3), rN], '-m', LW, 2)
   axis(EJES)
   TEXT = num2str(t_c(n));
   title(['Nondimensional time t = ', TEXT], FN, HNB, FS, 20)
   ylabel('Nondimensional Car Density.', FN, HNB, FS, 18)
   xlabel('Nondimensional distance on road.', FN, HNB, FS, 18)
   hold off
   pause(0.2)
   pause(0.2)
end
%
if nargout == 0
   clear x_c rho_c c_c t_c
end
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Copyright 1999-2008 Massachusetts Institute of Technology
% Version 01 by Rodolfo R. Rosales  03-25-1999.
% Update     by Rodolfo R. Rosales  02-05-2005.
% Update     by Rodolfo R. Rosales  02-15-2008.
% 
% Permission is hereby granted, without payment, to copy this software
% and its documentation, if any,  for non-profit academic and research
% purposes,  provided that the above copyright notice, this paragraph,
% and the following three paragraphs appear in all copies of this
% software. Use of this software constitutes acceptance of these terms
% and conditions.
%
% IN NO EVENT SHALL MIT, OR THE AUTHOR, BE LIABLE TO ANY PARTY FOR
% DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
% ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
% IF MIT, OR THE AUTHOR, HAVE BEEN ADVISED OF THE POSSIBILITY OF SUCH
% DAMAGE.
%
% MIT, AND THE AUTHOR, SPECIFICALLY DISCLAIM ANY EXPRESS OR IMPLIED
% WARRANTIES INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
% MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, AND
% NON-INFRINGEMENT.
%
% THIS SOFTWARE IS PROVIDED "AS IS." MIT, OR THE AUTHOR, HAVE NO
% OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS,
% OR MODIFICATIONS.
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% EOF
